package com.atlassian.vcache.internal.legacy;

import com.atlassian.cache.Cache;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.internal.core.service.LocalCacheUtils;
import com.atlassian.vcache.internal.core.service.VCacheLock;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link JvmCache} that backs onto a legacy {@link Cache}.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0.0
 */
class LegacyJvmCache<K, V> implements JvmCache<K, V> {
    private final Cache<K, V> delegate;
    private final VCacheLock globalLock;

    LegacyJvmCache(Cache<K, V> delegate, Duration lockTimeout) {
        this.delegate = requireNonNull(delegate);
        this.globalLock = new VCacheLock(delegate.getName(), lockTimeout);
    }

    @Override
    public Set<K> getKeys() {
        return globalLock.withLock(() -> new HashSet<>(delegate.getKeys()));
    }

    @Override
    public Optional<V> get(K key) {
        return globalLock.withLock(() -> Optional.ofNullable(delegate.get(key)));
    }

    @Override
    public V get(K key, Supplier<? extends V> supplier) {
        // Look for the existing value
        final Optional<V> current = get(requireNonNull(key));

        return current.orElseGet(() -> {
            // Calculate the missing value
            final V candidateValue = requireNonNull(supplier.get());

            // Now return either candidate value because one does not exist, otherwise the value that beat us to be inserted
            return globalLock.withLock(() -> delegate.get(key, () -> candidateValue));
        });
    }

    @SafeVarargs
    @Override
    public final Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, K... keys) {
        // This method needs to be final to prevent subclasses bypassing the locking enforced by this class.
        return getBulk(factory, Arrays.asList(keys));
    }

    @Override
    public Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, Iterable<K> keys) {
        return globalLock.withLock(() -> LocalCacheUtils.getBulk(
                factory,
                keys,
                this::get,
                args -> putIfAbsent(args.key, args.value),
                globalLock));
    }

    @Override
    public void put(K key, V value) {
        globalLock.withLock(() -> delegate.put(key, value));
    }

    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        return globalLock.withLock(() -> Optional.ofNullable(delegate.putIfAbsent(key, value)));
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        return globalLock.withLock(() -> delegate.replace(key, currentValue, newValue));
    }

    @Override
    public boolean removeIf(K key, V value) {
        return globalLock.withLock(() -> delegate.remove(key, value));
    }

    @Override
    public void remove(K key) {
        globalLock.withLock(() -> delegate.remove(key));
    }

    @Override
    public void removeAll() {
        globalLock.withLock(delegate::removeAll);
    }

    @Override
    public String getName() {
        return delegate.getName();
    }
}
