package com.atlassian.vcache.internal.guava;

import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.PutPolicy;
import com.atlassian.vcache.internal.MetricLabel;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.cas.IdentifiedData;
import com.atlassian.vcache.internal.core.metrics.CacheType;
import com.atlassian.vcache.internal.core.metrics.MetricsRecorder;
import com.atlassian.vcache.internal.core.service.AbstractExternalCacheRequestContext;
import com.atlassian.vcache.internal.core.service.AbstractStableReadExternalCache;
import com.atlassian.vcache.internal.core.service.UnversionedExternalCacheRequestContext;
import com.google.common.cache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.core.cas.IdentifiedUtils.marshall;
import static com.atlassian.vcache.internal.core.cas.IdentifiedUtils.unmarshall;
import static java.util.Objects.requireNonNull;

/**
 * Guava implementation of {@link com.atlassian.vcache.StableReadExternalCache}.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
public class GuavaStableReadExternalCache<V>
        extends AbstractStableReadExternalCache<V> {
    private static final Logger log = LoggerFactory.getLogger(GuavaStableReadExternalCache.class);

    private final Cache<String, IdentifiedData> delegate;
    private final Supplier<RequestContext> contextSupplier;
    private final ExternalCacheKeyGenerator keyGenerator;
    private final Optional<MarshallingPair<V>> valueMarshalling;

    public GuavaStableReadExternalCache(
            String name,
            Cache<String, IdentifiedData> delegate,
            Supplier<RequestContext> contextSupplier,
            ExternalCacheKeyGenerator keyGenerator,
            Optional<MarshallingPair<V>> valueMarshalling,
            MetricsRecorder metricsRecorder,
            Duration lockTimeout) {
        super(name, metricsRecorder, lockTimeout, (n, ex) -> {});
        this.contextSupplier = requireNonNull(contextSupplier);
        this.keyGenerator = requireNonNull(keyGenerator);
        this.valueMarshalling = requireNonNull(valueMarshalling);
        this.delegate = requireNonNull(delegate);
    }

    @Override
    protected boolean internalPut(String internalKey, V value, PutPolicy policy) {
        final IdentifiedData identifiedData = marshall(value, valueMarshalling);
        final String externalKey = ensureCacheContext().externalEntryKeyFor(internalKey);
        return GuavaUtils.directPut(externalKey, identifiedData, policy, delegate);
    }

    @Override
    protected void internalRemove(Iterable<String> internalKeys) {
        // There is no bulk delete in the api, so need to remove each one
        final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        for (String key : internalKeys) {
            delegate.asMap().remove(cacheContext.externalEntryKeyFor(key));
            cacheContext.recordValue(key, Optional.empty());
        }
    }

    @Override
    protected void internalRemoveAll() {
        delegate.asMap().clear();
    }

    @Override
    protected Logger getLogger() {
        return log;
    }

    @Override
    protected AbstractExternalCacheRequestContext<V> ensureCacheContext() {
        final RequestContext requestContext = contextSupplier.get();

        return requestContext.computeIfAbsent(this, () -> {
            log.trace("Cache {}: Setting up a new context", getName());
            return new UnversionedExternalCacheRequestContext<>(
                    keyGenerator,
                    getName(),
                    requestContext::partitionIdentifier,
                    lockTimeout);
        });
    }

    @Override
    protected V handleCreation(String internalKey, V candidateValue)
            throws ExecutionException, InterruptedException {
        final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
        final IdentifiedData candidateIdentifiedData = marshall(candidateValue, valueMarshalling);

        final String externalKey = cacheContext.externalEntryKeyFor(internalKey);
        metricsRecorder.record(name, CacheType.EXTERNAL, MetricLabel.NUMBER_OF_REMOTE_GET, 1);
        final Optional<V> otherAddedValue =
                unmarshall(delegate.asMap().putIfAbsent(externalKey, candidateIdentifiedData), valueMarshalling);

        if (otherAddedValue.isPresent()) {
            getLogger().info("Cache {}, unable to add candidate for key {}, use what was added", name, internalKey);
            // Record as if doing another remote call
            metricsRecorder.record(name, CacheType.EXTERNAL, MetricLabel.NUMBER_OF_REMOTE_GET, 1);
            return otherAddedValue.get();
        }

        return candidateValue;
    }

    @Override
    protected final ExternalCacheException mapException(Exception ex) {
        return GuavaUtils.mapException(ex);
    }

    @Override
    protected final Optional<V> directGet(String externalKey) {
        return unmarshall(delegate.getIfPresent(externalKey), valueMarshalling);
    }

    @Override
    protected final Map<String, Optional<V>> directGetBulk(Set<String> externalKeys) {
        return GuavaUtils.directGetBulk(externalKeys, delegate, valueMarshalling);
    }
}
