package com.atlassian.vcache.internal.guava;

import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.Sha1ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import com.atlassian.vcache.internal.core.service.AbstractVCacheService;
import com.atlassian.vcache.internal.core.service.GuavaJvmCache;
import com.google.common.cache.Cache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Main service that is backed by <tt>Guava</tt>.
 *
 * @since 1.0.0
 */
public class GuavaVCacheService extends AbstractVCacheService {
    private static final Logger log = LoggerFactory.getLogger(GuavaVCacheService.class);

    private final GuavaServiceSettings serviceSettings;
    // Used to keep track of the underlying Guava caches used. This is to simulate the behaviour
    // of the real external caches.
    private final Map<String, Cache> delegatedCaches = new ConcurrentHashMap<>();

    public GuavaVCacheService(
            String productIdentifier,
            Supplier<RequestContext> threadLocalContextSupplier,
            Supplier<RequestContext> workContextContextSupplier,
            VCacheSettingsDefaultsProvider defaultsProvider,
            VCacheCreationHandler creationHandler,
            MetricsCollector metricsCollector,
            GuavaServiceSettings serviceSettings,
            BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        super(threadLocalContextSupplier,
                workContextContextSupplier,
                defaultsProvider,
                creationHandler,
                metricsCollector,
                new Sha1ExternalCacheKeyGenerator(productIdentifier),
                begunTransactionalActivityHandler,
                serviceSettings.getLockTimeout());
        this.serviceSettings = requireNonNull(serviceSettings);
    }

    public GuavaVCacheService(
            Supplier<RequestContext> threadLocalContextSupplier,
            Supplier<RequestContext> workContextContextSupplier,
            VCacheSettingsDefaultsProvider defaultsProvider,
            VCacheCreationHandler creationHandler,
            MetricsCollector metricsCollector,
            ExternalCacheKeyGenerator externalCacheKeyGenerator,
            GuavaServiceSettings serviceSettings,
            BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        super(threadLocalContextSupplier,
                workContextContextSupplier,
                defaultsProvider,
                creationHandler,
                metricsCollector,
                externalCacheKeyGenerator,
                begunTransactionalActivityHandler,
                serviceSettings.getLockTimeout());
        this.serviceSettings = requireNonNull(serviceSettings);
    }

    @Override
    protected Logger log() {
        return log;
    }

    @Override
    protected <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings) {
        return new GuavaJvmCache<>(name, settings, lockTimeout);
    }

    @Override
    protected <V> TransactionalExternalCache<V> createTransactionalExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        return new GuavaTransactionalExternalCache<>(
                name,
                obtainDelegate(name, settings),
                threadLocalContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                transactionControlManager,
                metricsCollector,
                serviceSettings.getLockTimeout());
    }

    @Override
    protected <V> StableReadExternalCache<V> createStableReadExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        return new GuavaStableReadExternalCache<>(
                name,
                obtainDelegate(name, settings),
                workContextContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                metricsCollector,
                serviceSettings.getLockTimeout());
    }

    @Override
    protected <V> DirectExternalCache<V> createDirectExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable) {
        return new GuavaDirectExternalCache<>(
                name,
                obtainDelegate(name, settings),
                workContextContextSupplier,
                externalCacheKeyGenerator,
                (serviceSettings.isSerializationHack() && valueSerializable) ? Optional.empty() : Optional.of(valueMarshalling),
                serviceSettings.getLockTimeout());
    }

    private <V> Cache<String, V> obtainDelegate(String name, ExternalCacheSettings settings) {
        return delegatedCaches.computeIfAbsent(name, k -> GuavaUtils.buildDelegate(settings));
    }
}
