package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.VCacheException;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.UncheckedExecutionException;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Implementation of the {@link JvmCache} that uses Guava.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
public class GuavaJvmCache<K, V> implements JvmCache<K, V> {
    private final String name;
    private final VCacheLock globalLock;
    private final Cache<K, V> delegate;

    /**
     * Creates an instance. <b>NOTE:</b> all the supplied settings must be set.
     *
     * @param name     the name of the cache
     * @param settings the settings for the cache.
     */
    public GuavaJvmCache(String name, JvmCacheSettings settings, Duration lockTimeout) {
        this.name = requireValidCacheName(name);
        this.globalLock = new VCacheLock(name, lockTimeout);
        //noinspection OptionalGetWithoutIsPresent
        this.delegate = CacheBuilder.newBuilder()
                .maximumSize(settings.getMaxEntries().get())
                .expireAfterWrite(settings.getDefaultTtl().get().toMillis(), TimeUnit.MILLISECONDS)
                .build();
    }

    @Override
    public Set<K> getKeys() {
        return globalLock.withLock(() -> new HashSet<>(delegate.asMap().keySet()));
    }

    @Override
    public Optional<V> get(K key) {
        return globalLock.withLock(() -> Optional.ofNullable(delegate.getIfPresent(key)));
    }

    @Override
    public V get(K key, Supplier<? extends V> supplier) {
        // Look for the existing value
        final Optional<V> current = get(requireNonNull(key));

        return current.orElseGet(() -> {
            // Calculate the missing value
            final V candidateValue = requireNonNull(supplier.get());

            // Now return either candidate value because one does not exist, otherwise the value that beat us to be inserted
            return globalLock.withLock(() -> {
                try {
                    return delegate.get(key, () -> candidateValue);
                } catch (UncheckedExecutionException | ExecutionException e) {
                    throw new VCacheException("Internal Guava failure", e);
                }
            });
        });
    }

    @SafeVarargs
    @Override
    public final Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, K... keys) {
        // This method needs to be final to prevent subclasses bypassing the locking enforced by this class.
        return getBulk(factory, Arrays.asList(keys));
    }

    @Override
    public Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, Iterable<K> keys) {
        return globalLock.withLock(() -> LocalCacheUtils.getBulk(
                factory,
                keys,
                this::get,
                args -> putIfAbsent(args.key, args.value),
                globalLock));
    }

    @Override
    public void put(K key, V value) {
        globalLock.withLock(() -> delegate.put(requireNonNull(key), requireNonNull(value)));
    }

    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        return globalLock.withLock(() -> Optional.ofNullable(delegate.asMap().putIfAbsent(requireNonNull(key), requireNonNull(value))));
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        return globalLock.withLock(() -> delegate.asMap().replace(requireNonNull(key), requireNonNull(currentValue), requireNonNull(newValue)));
    }

    @Override
    public void remove(K key) {
        globalLock.withLock(() -> delegate.invalidate(key));
    }

    @Override
    public void removeAll() {
        globalLock.withLock((Runnable) delegate::invalidateAll);
    }

    @Override
    public boolean removeIf(K key, V value) {
        return globalLock.withLock(() -> delegate.asMap().remove(key, value));
    }

    @Override
    public String getName() {
        return name;
    }
}
