package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.RequestCache;
import com.atlassian.vcache.internal.RequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link RequestCache} that uses a delegate {@link HashMap}
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
class DefaultRequestCache<K, V> implements RequestCache<K, V> {
    public static final Logger log = LoggerFactory.getLogger(DefaultRequestCache.class);

    private final String name;
    private final Supplier<RequestContext> contextSupplier;
    private final Duration lockTimeout;

    DefaultRequestCache(String name, Supplier<RequestContext> contextSupplier, Duration lockTimeout) {
        this.name = requireValidCacheName(name);
        this.contextSupplier = requireNonNull(contextSupplier);
        this.lockTimeout = requireNonNull(lockTimeout);
    }

    @Override
    public Optional<V> get(K key) {
        return withLock(kvMap -> Optional.ofNullable(kvMap.get(key)));
    }

    @Override
    public V get(K key, Supplier<? extends V> supplier) {
        // Look for the existing value
        final Optional<V> current = get(key);

        return current.orElseGet(() -> {
            // Calculate the missing value
            final V candidateValue = requireNonNull(supplier.get());

            // Now return either candidate value because one does not exist, otherwise the value that beat us to be inserted
            return withLock(kvMap -> {
                final V existing = kvMap.putIfAbsent(requireNonNull(key), candidateValue);
                return (existing == null) ? candidateValue : existing;
            });
        });
    }

    @SafeVarargs
    @Override
    public final Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, K... keys) {
        // This method needs to be final to prevent subclasses bypassing the locking enforced by this class.
        return getBulk(factory, Arrays.asList(keys));
    }

    @Override
    public Map<K, V> getBulk(Function<Set<K>, Map<K, V>> factory, Iterable<K> keys) {
        return withLock(kvMap -> LocalCacheUtils.getBulk(
                factory,
                keys,
                this::get,
                args -> putIfAbsent(args.key, args.value),
                ensureDelegate().lock));
    }

    @Override
    public void put(K key, V value) {
        withLock(kvMap -> kvMap.put(requireNonNull(key), requireNonNull(value)));
    }

    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        return withLock(kvMap -> Optional.ofNullable(
                kvMap.putIfAbsent(requireNonNull(key), requireNonNull(value))));
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        return withLock(kvMap ->
                kvMap.replace(requireNonNull(key), requireNonNull(currentValue), requireNonNull(newValue)));
    }

    @Override
    public boolean removeIf(K key, V value) {
        return withLock(kvMap -> kvMap.remove(requireNonNull(key), requireNonNull(value)));
    }

    @Override
    public void remove(K key) {
        withLock(kvMap -> kvMap.remove(key));
    }

    @Override
    public void removeAll() {
        withLock(kvMap -> {
                kvMap.clear();
                // Dodgy 'return false' to simulate returning a value.
                return false;
            });
    }

    @Override
    public String getName() {
        return name;
    }

    private MapAndLock ensureDelegate() {
        final RequestContext requestContext = contextSupplier.get();
        return requestContext.computeIfAbsent(this, MapAndLock::new);
    }

    private <R> R withLock(Function<Map<K, V>, R> fn) {
        final MapAndLock mal = ensureDelegate();

        return mal.lock.withLock(() -> fn.apply(mal.map));
    }

    private class MapAndLock {
        final Map<K, V> map = new HashMap<>();
        final VCacheLock lock = new VCacheLock(name, lockTimeout);
    }
}
