package com.atlassian.vcache.internal.core.service;

import com.atlassian.marshalling.api.MarshallingException;
import com.atlassian.vcache.ExternalCache;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.internal.ExternalCacheExceptionListener;
import com.atlassian.vcache.internal.core.VCacheCoreUtils;
import org.slf4j.Logger;

import java.time.Duration;
import java.util.concurrent.Callable;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;

import static com.atlassian.vcache.ExternalCacheException.Reason.MARSHALLER_FAILURE;
import static com.atlassian.vcache.ExternalCacheException.Reason.UNCLASSIFIED_FAILURE;
import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static com.atlassian.vcache.internal.core.VCacheCoreUtils.failed;
import static java.util.Objects.requireNonNull;

/**
 * Provides common operations for {@link ExternalCache} instances.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
public abstract class AbstractExternalCache<V> implements ExternalCache<V> {
    protected final String name;
    protected final Duration lockTimeout;
    private ExternalCacheExceptionListener externalCacheExceptionListener;

    protected AbstractExternalCache(String name,
                                    Duration lockTimeout,
                                    ExternalCacheExceptionListener externalCacheExceptionListener) {
        this.name = requireValidCacheName(name);
        this.lockTimeout = requireNonNull(lockTimeout);
        this.externalCacheExceptionListener = requireNonNull(externalCacheExceptionListener);
    }

    /**
     * Returns the cache context for the current request.
     *
     * @return the cache context for the current request.
     */
    protected abstract AbstractExternalCacheRequestContext<V> ensureCacheContext();

    /**
     * Returns the logging instance for the implementation class.
     *
     * @return the logging instance for the implementation class.
     */
    protected abstract Logger getLogger();

    /**
     * Maps a generic {@link Exception} to a corresponding {@link ExternalCacheException}.
     */
    protected abstract ExternalCacheException mapException(Exception ex);

    @Override
    public final String getName() {
        return name;
    }

    /**
     * Performs a transaction and handles all the standard exceptions that may occur.
     *
     * @param txn the transaction the perform
     * @param <T> the return type
     * @return A {@link CompletionStage} representing the outcome of performing the transaction.
     */
    protected <T> CompletionStage<T> perform(Callable<T> txn) {
        return perform(txn, (i) -> {
        });
    }

    /**
     * Performs a transaction and handles all the standard exceptions that may occur.
     *
     * @param txn            the transaction the perform
     * @param successHandler called if the <tt>txn</tt> is successful
     * @param <T>            the return type
     * @return A {@link CompletionStage} representing the outcome of performing the transaction.
     */
    protected <T> CompletionStage<T> perform(Callable<T> txn, Consumer<T> successHandler) {
        final ExternalCacheException exception;
        try {
            final T outcome = txn.call();
            successHandler.accept(outcome);
            return VCacheCoreUtils.successful(outcome);
        } catch (MarshallingException ex) {
            exception = new ExternalCacheException(MARSHALLER_FAILURE, ex);
        } catch (ExecutionException | InterruptedException ex) {
            exception = new ExternalCacheException(UNCLASSIFIED_FAILURE, ex);
        } catch (ExternalCacheException ece) {
            exception = ece;
        } catch (Exception ex) {
            exception = mapException(ex);
        }

        externalCacheExceptionListener.onThrow(getName(), exception);
        return failed(new CompletableFuture<>(), exception);
    }
}
