package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.marshalling.api.Marshaller;
import com.atlassian.marshalling.api.MarshallingException;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_BYTES_MARSHALLED;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_MARSHALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_MARSHALL_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link Marshaller} that records metrics.
 *
 * @param <T> the value type
 * @since 1.0.0
 */
class TimedMarshaller<T> implements Marshaller<T> {
    private final Marshaller<T> delegate;
    private final MetricsRecorder metricsRecorder;
    private final String cacheName;

    TimedMarshaller(Marshaller<T> delegate, MetricsRecorder metricsRecorder, String cacheName) {
        this.delegate = requireNonNull(delegate);
        this.metricsRecorder = requireNonNull(metricsRecorder);
        this.cacheName = requireNonNull(cacheName);
    }

    @Override
    public byte[] marshallToBytes(T obj) throws MarshallingException {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(cacheName, EXTERNAL, TIMED_MARSHALL_CALL, t))) {
            final byte[] result = delegate.marshallToBytes(obj);
            metricsRecorder.record(cacheName, EXTERNAL, NUMBER_OF_BYTES_MARSHALLED, result.length);
            return result;
        } catch (MarshallingException me) {
            metricsRecorder.record(cacheName, EXTERNAL, NUMBER_OF_FAILED_MARSHALL, 1);
            throw me;
        }
    }
}
