package com.atlassian.vcache.internal.core.cas;

import com.atlassian.marshalling.api.MarshallingException;
import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.CasIdentifier;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.IdentifiedValue;
import com.atlassian.vcache.internal.core.DefaultIdentifiedValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Optional;

import static com.atlassian.vcache.ExternalCacheException.Reason.MARSHALLER_FAILURE;
import static com.atlassian.vcache.ExternalCacheException.Reason.UNCLASSIFIED_FAILURE;
import static java.util.Objects.requireNonNull;

/**
 * Contains common methods for dealing with {@link IdentifiedData}.
 *
 * @since 1.1.0
 */
public class IdentifiedUtils {
    private static final Logger log = LoggerFactory.getLogger(IdentifiedUtils.class);

    public static <V> IdentifiedData marshall(V data, Optional<MarshallingPair<V>> valueMarshalling) throws ExternalCacheException {
        requireNonNull(data);
        try {
            return valueMarshalling.isPresent()
                    ? new IdentifiedDataBytes(valueMarshalling.get().getMarshaller().marshallToBytes(data))
                    : new IdentifiedDataSerializable((Serializable) data);
        } catch (MarshallingException e) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, e);
        }
    }

    @SuppressWarnings("unchecked")
    public static <V> Optional<V> unmarshall(@Nullable IdentifiedData idata, Optional<MarshallingPair<V>> valueMarshalling)
            throws ExternalCacheException {
        if (idata == null) {
            return Optional.empty();
        }

        try {
            return valueMarshalling.isPresent()
                    ? Optional.of(valueMarshalling.get().getUnmarshaller().unmarshallFrom(((IdentifiedDataBytes) idata).getBytes()))
                    : Optional.of((V) ((IdentifiedDataSerializable) idata).getObject());
        } catch (MarshallingException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    @SuppressWarnings("unchecked")
    public static <V> Optional<IdentifiedValue<V>> unmarshallIdentified(
            @Nullable IdentifiedData idata, Optional<MarshallingPair<V>> valueMarshalling) {
        if (idata == null) {
            return Optional.empty();
        }

        try {
            final V value = valueMarshalling.isPresent()
                    ? valueMarshalling.get().getUnmarshaller().unmarshallFrom(((IdentifiedDataBytes) idata).getBytes())
                    : (V) ((IdentifiedDataSerializable) idata).getObject();
            return Optional.of(new DefaultIdentifiedValue<>(idata, value));
        } catch (MarshallingException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    public static IdentifiedData safeCast(CasIdentifier casId) {
        if (casId instanceof IdentifiedData) {
            return (IdentifiedData) casId;
        }

        log.warn("Passed an unknown CasIdentifier instance of class {}.", casId.getClass().getName());
        throw new ExternalCacheException(UNCLASSIFIED_FAILURE);
    }
}
