package com.atlassian.vcache.internal.core;

import com.google.common.hash.Hashing;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.Base64;

/**
 * Implementation of {@link ExternalCacheKeyGenerator} that encodes using the <tt>SHA-1</tt> hashing algorithm.
 *
 * @since 1.0.0
 */
public class Sha1ExternalCacheKeyGenerator extends ExternalCacheKeyGenerator {
    private static final Charset DIGEST_CHARSET = StandardCharsets.UTF_8;

    public Sha1ExternalCacheKeyGenerator(String productIdentifier) {
        super(productIdentifier);
    }

    @Override
    protected String encode(String plain) {
        /*
         * We are not using MessageDigest#getInstance because under high load this becomes a contention. This is a known issue in Java
         * documented here: https://bugs.java.com/bugdatabase/view_bug.do?bug_id=7092821
         * We are using the Guava implementation instead as they have optimized this bit.
         */
        final byte[] digest = Hashing.sha1().hashString(plain, DIGEST_CHARSET).asBytes();
        return Base64.getEncoder().encodeToString(digest);
    }
}
