package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.VCacheException;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.util.concurrent.UncheckedExecutionException;

import javax.annotation.Nonnull;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Implementation of the {@link JvmCache} that uses Guava.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
public class GuavaJvmCache<K, V> extends AbstractLockingJvmCache<K, V> {
    private final String name;
    private final Cache<K, V> delegate;

    /**
     * Creates an instance. <b>NOTE:</b> all the supplied settings must be set.
     *
     * @param name     the name of the cache
     * @param settings the settings for the cache.
     */
    public GuavaJvmCache(String name, JvmCacheSettings settings) {
        this.name = requireValidCacheName(name);
        this.delegate = CacheBuilder.newBuilder()
                .maximumSize(settings.getMaxEntries().get())
                .expireAfterAccess(settings.getDefaultTtl().get().toMillis(), TimeUnit.MILLISECONDS)
                .build();
    }

    @Nonnull
    @Override
    protected V decoratedGet(K key, Supplier<? extends V> supplier) {
        try {
            return delegate.get(requireNonNull(key), supplier::get);
        } catch (UncheckedExecutionException | ExecutionException e) {
            throw new VCacheException("Supplier failed", e);
        }
    }

    @Override
    protected void decoratedRemove(K key) {
        delegate.invalidate(key);
    }

    @Override
    protected void decoratedRemoveAll() {
        delegate.invalidateAll();
    }

    @Nonnull
    @Override
    public Set<K> getKeys() {
        return new HashSet<>(delegate.asMap().keySet());
    }

    @Nonnull
    @Override
    public Optional<V> get(K key) {
        return Optional.ofNullable(delegate.getIfPresent(key));
    }

    @Override
    public void put(K key, V value) {
        delegate.put(requireNonNull(key), requireNonNull(value));
    }

    @Nonnull
    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        return Optional.ofNullable(delegate.asMap().putIfAbsent(requireNonNull(key), requireNonNull(value)));
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        return delegate.asMap().replace(requireNonNull(key), requireNonNull(currentValue), requireNonNull(newValue));
    }

    @Override
    public boolean removeIf(K key, V value) {
        return delegate.asMap().remove(key, value);
    }

    @Nonnull
    @Override
    public String getName() {
        return name;
    }
}
