package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCache;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.RequestCache;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.VCacheFactory;
import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.ExternalCacheDetails;
import com.atlassian.vcache.internal.JvmCacheDetails;
import com.atlassian.vcache.internal.RequestCacheDetails;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.RequestMetrics;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheLifecycleManager;
import com.atlassian.vcache.internal.VCacheManagement;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.DefaultExternalCacheDetails;
import com.atlassian.vcache.internal.core.DefaultJvmCacheDetails;
import com.atlassian.vcache.internal.core.DefaultRequestCache;
import com.atlassian.vcache.internal.core.DefaultRequestCacheDetails;
import com.atlassian.vcache.internal.core.DefaultTransactionControlManager;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.TransactionControlManager;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import org.slf4j.Logger;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Base implementation of a service.
 *
 * @since 1.0.0
 */
public abstract class AbstractVCacheService implements VCacheFactory, VCacheManagement, VCacheLifecycleManager {
    // List of Marshaller class names that support Serializable values. This is a slightly
    // dodgy approach, but the rationale is to not be forced to make implementations
    // visible, or to make it an attribute of a Marshaller. The fact is, this is a temporary
    // hack for internal use only.
    private static final Set<String> SERIALIZABLE_MARSHALLER_CLASS_NAMES =
            Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
                    "com.atlassian.vcache.marshallers.StringMarshaller",
                    "com.atlassian.vcache.marshallers.JavaSerializationMarshaller")));

    protected final Supplier<RequestContext> contextSupplier;
    protected final TransactionControlManager transactionControlManager;
    protected final ExternalCacheKeyGenerator externalCacheKeyGenerator;

    private final VCacheSettingsDefaultsProvider defaultsProvider;
    private final VCacheCreationHandler creationHandler;
    private final MetricsCollector metricsCollector;

    private final ConcurrentHashMap<String, JvmCache> jvmCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, JvmCacheDetails> jvmCacheDetailsMap = new ConcurrentHashMap<>();

    private final ConcurrentHashMap<String, RequestCache> requestCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, RequestCacheDetails> requestCacheDetailsMap = new ConcurrentHashMap<>();

    private final ConcurrentHashMap<String, ExternalCache> externalCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, ExternalCacheDetails> externalCacheDetailsMap = new ConcurrentHashMap<>();

    public AbstractVCacheService(Supplier<RequestContext> contextSupplier,
                                 VCacheSettingsDefaultsProvider defaultsProvider,
                                 VCacheCreationHandler creationHandler,
                                 MetricsCollector metricsCollector,
                                 ExternalCacheKeyGenerator externalCacheKeyGenerator,
                                 BegunTransactionalActivityHandler begunTransactionalActivityHandler) {
        this.contextSupplier = requireNonNull(contextSupplier);
        this.defaultsProvider = requireNonNull(defaultsProvider);
        this.creationHandler = requireNonNull(creationHandler);
        this.metricsCollector = requireNonNull(metricsCollector);
        this.transactionControlManager = new DefaultTransactionControlManager(metricsCollector, begunTransactionalActivityHandler);
        this.externalCacheKeyGenerator = requireNonNull(externalCacheKeyGenerator);
    }

    /**
     * Returns the logger for the implementation service.
     *
     * @return the logger for the implementation service.
     */
    @Nonnull
    protected abstract Logger log();

    /**
     * Creates a {@link JvmCache} with the supplied parameters.
     *
     * @param name     the name of the cache
     * @param settings the setting for the cache
     * @param <K>      the key type
     * @param <V>      the value type
     * @return a freshly minted {@link JvmCache}
     */
    @Nonnull
    protected abstract <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings);

    /**
     * Creates a {@link TransactionalExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshaller   the marshaller for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link TransactionalExternalCache}
     */
    @Nonnull
    protected abstract <V> TransactionalExternalCache<V> createTransactionalExternalCache(
            String name, ExternalCacheSettings settings, Marshaller<V> valueMarshaller, boolean valueSerializable);

    /**
     * Creates a {@link StableReadExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshaller   the marshaller for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link StableReadExternalCache}
     */
    @Nonnull
    protected abstract <V> StableReadExternalCache<V> createStableReadExternalCache(
            String name, ExternalCacheSettings settings, Marshaller<V> valueMarshaller, boolean valueSerializable);

    /**
     * Creates a {@link DirectExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshaller   the marshaller for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link DirectExternalCache}
     */
    @Nonnull
    protected abstract <V> DirectExternalCache<V> createDirectExternalCache(
            String name, ExternalCacheSettings settings, Marshaller<V> valueMarshaller, boolean valueSerializable);

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public <K, V> JvmCache<K, V> getJvmCache(String name, JvmCacheSettings settings) {
        return jvmCacheInstancesMap.computeIfAbsent(requireValidCacheName(name), s -> {
            log().trace("Cache {}: creating the instance", name);
            final JvmCacheSettings candidateSettings = defaultsProvider.getJvmDefaults(name).override(settings);
            final JvmCacheSettings finalSettings =
                    creationHandler.jvmCacheCreation(new DefaultJvmCacheDetails(name, candidateSettings));
            jvmCacheDetailsMap.put(name, new DefaultJvmCacheDetails(name, finalSettings));

            return metricsCollector.wrap(createJvmCache(name, finalSettings));
        });
    }

    @Nonnull
    @Override
    @SuppressWarnings("unchecked")
    public <K, V> RequestCache<K, V> getRequestCache(String name) {
        return requestCacheInstancesMap.computeIfAbsent(requireValidCacheName(name), s -> {
            log().trace("Cache {}: creating the instance", name);
            creationHandler.requestCacheCreation(name);
            requestCacheDetailsMap.put(name, new DefaultRequestCacheDetails(name));
            return metricsCollector.wrap(new DefaultRequestCache(name, contextSupplier));
        });
    }

    @Nonnull
    @Override
    public <V> TransactionalExternalCache<V> getTransactionalExternalCache(
            String name, Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final Marshaller<V> wrappedMarshaller = metricsCollector.wrap(valueMarshaller, name);
        final TransactionalExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.FULLY, settings, finalSettings ->
                        createTransactionalExternalCache(
                                name, finalSettings, wrappedMarshaller, isValueSerizable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Nonnull
    @Override
    public <V> StableReadExternalCache<V> getStableReadExternalCache(
            String name, Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final Marshaller<V> wrappedMarshaller = metricsCollector.wrap(valueMarshaller, name);
        final StableReadExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.READ_ONLY, settings, finalSettings ->
                        createStableReadExternalCache(
                                name, finalSettings, wrappedMarshaller, isValueSerizable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Nonnull
    @Override
    public <V> DirectExternalCache<V> getDirectExternalCache(
            String name, Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final Marshaller<V> wrappedMarshaller = metricsCollector.wrap(valueMarshaller, name);
        final DirectExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.NEVER, settings, finalSettings ->
                        createDirectExternalCache(
                                name, finalSettings, wrappedMarshaller, isValueSerizable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public void transactionSync(RequestContext context) {
        transactionControlManager.syncAll(context);
    }

    @Nonnull
    @Override
    public Set<String> transactionDiscard(RequestContext context) {
        return transactionControlManager.discardAll(context);
    }

    @Nonnull
    @Override
    public RequestMetrics metrics(RequestContext context) {
        return metricsCollector.obtainRequestMetrics(context);
    }

    @Nonnull
    @Override
    public Map<String, JvmCacheDetails> allJvmCacheDetails() {
        final Map<String, JvmCacheDetails> result = new HashMap<>();
        result.putAll(jvmCacheDetailsMap);
        return result;
    }

    @Nonnull
    @Override
    public Map<String, RequestCacheDetails> allRequestCacheDetails() {
        final Map<String, RequestCacheDetails> result = new HashMap<>();
        result.putAll(requestCacheDetailsMap);
        return result;
    }

    @Nonnull
    @Override
    public Map<String, ExternalCacheDetails> allExternalCacheDetails() {
        final Map<String, ExternalCacheDetails> result = new HashMap<>();
        result.putAll(externalCacheDetailsMap);
        return result;
    }

    private <V> boolean isValueSerizable(Marshaller<V> valueMarshaller) {
        return SERIALIZABLE_MARSHALLER_CLASS_NAMES.contains(valueMarshaller.getClass().getName());
    }

    @SuppressWarnings("unchecked")
    private <C extends ExternalCache> C obtainCache(
            String name,
            ExternalCacheDetails.BufferPolicy policy,
            ExternalCacheSettings settings,
            Function<ExternalCacheSettings, C> factory) {
        // Retrieve or create the cache. Then verify that the returned cache is of the correct type.
        // There could be a race condition where by two threads create a cache of the same name
        // with different types.
        final ExternalCache candidateCache = externalCacheInstancesMap.computeIfAbsent(
                requireValidCacheName(name), s -> {
                    log().trace("Cache {}: creating the instance with policy {}", name, policy);
                    final ExternalCacheSettings candidateSettings =
                            defaultsProvider.getExternalDefaults(name).override(settings);
                    final ExternalCacheSettings finalSettings = creationHandler.externalCacheCreation(
                            new DefaultExternalCacheDetails(name, policy, candidateSettings));
                    externalCacheDetailsMap.put(
                            name, new DefaultExternalCacheDetails(name, policy, finalSettings));
                    return factory.apply(finalSettings);
                });

        if (externalCacheDetailsMap.get(name).getPolicy() != policy) {
            log().warn("Cache {}: unable to create cache with policy {}, as one already configured with policy {}",
                    name, ExternalCacheDetails.BufferPolicy.READ_ONLY, externalCacheDetailsMap.get(name).getPolicy());
            throw new ExternalCacheException(ExternalCacheException.Reason.CREATION_FAILURE);
        }

        return (C) candidateCache;
    }
}
