package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.MarshallerException;

import javax.annotation.Nonnull;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import static com.atlassian.vcache.internal.core.VCacheCoreUtils.isEmpty;

/**
 * Provides operations common to both {@link com.atlassian.vcache.StableReadExternalCache} and
 * {@link com.atlassian.vcache.TransactionalExternalCache} instances.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
public abstract class AbstractNonDirectExternalCache<V>
        extends AbstractExternalCache<V> {

    protected AbstractNonDirectExternalCache(String name) {
        super(name);
    }

    /**
     * Handles the creation of an entry, if required.
     *
     * @param internalKey the internal key for the entry.
     * @param supplier    called to create the value, if required
     * @return the value associated with the key.
     */
    @Nonnull
    protected abstract V handleCreation(String internalKey, Supplier<V> supplier)
            throws MarshallerException, ExecutionException, InterruptedException;

    /**
     * Handles the creation of a number of entries
     *
     * @param factory      called to create the required values.
     * @param externalKeys the external keys for the entries.
     * @return the values associated with the supplied keys (mapped on internal key)
     */
    @Nonnull
    protected abstract Map<String, V> handleCreation(
            Function<Set<String>, Map<String, V>> factory, Set<String> externalKeys)
            throws ExecutionException, InterruptedException;

    /**
     * Checks if a value is recorded for a specified internal key.
     */
    @Nonnull
    protected abstract Optional<Optional<V>> checkValueRecorded(String internalKey);

    /**
     * Checks for values recorded for specified internal keys.
     */
    @Nonnull
    protected abstract Map<String, Optional<V>> checkValuesRecorded(Iterable<String> internalKeys);

    /**
     * Performs a direct get operation against the external cache using the supplied external key.
     */
    @Nonnull
    protected abstract Optional<V> directGet(String externalKey);

    /**
     * Performs a direct bulk get operation against the external cache using the supplied external keys.
     */
    @Nonnull
    protected abstract Map<String, Optional<V>> directGetBulk(Set<String> externalKeys);

    @Nonnull
    @Override
    public final CompletionStage<Optional<V>> get(String internalKey) {
        return perform(() -> {
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();

            final Optional<Optional<V>> prior = checkValueRecorded(internalKey);

            return prior.orElseGet(() -> {
                final String externalKey = cacheContext.externalEntryKeyFor(internalKey);

                final Optional<V> result = directGet(externalKey);
                cacheContext.recordValue(internalKey, result);
                return result;
            });
        });
    }

    @Nonnull
    @Override
    public final CompletionStage<V> get(String internalKey, Supplier<V> supplier) {
        return perform(() -> {
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();

            final Optional<Optional<V>> prior = checkValueRecorded(internalKey);
            if (prior.isPresent()) {
                if (prior.get().isPresent()) {
                    return prior.get().get();
                } else {
                    getLogger().trace("Cache {}, creating candidate for key {}", name, internalKey);
                    return handleCreation(internalKey, supplier);
                }
            }

            // Either way, there was no prior value recorded, or it was blank. So need to create the value
            // and write it back.
            final String externalKey = cacheContext.externalEntryKeyFor(internalKey);
            final Optional<V> result = directGet(externalKey);
            if (result.isPresent()) {
                // A valid value exists in the external cache
                cacheContext.recordValue(internalKey, result);
                return result.get();
            }

            getLogger().trace("Cache {}, creating candidate for key {}", name, internalKey);
            return handleCreation(internalKey, supplier);
        });
    }

    @Nonnull
    @Override
    public final CompletionStage<Map<String, Optional<V>>> getBulk(Iterable<String> internalKeys) {
        return perform(() -> {
            if (isEmpty(internalKeys)) {
                return new HashMap<>();
            }

            // Get the recorded values first
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            final Map<String, Optional<V>> grandResult = checkValuesRecorded(internalKeys);

            // Calculate the externalKeys for the entries that are missing
            final Set<String> missingExternalKeys = StreamSupport.stream(internalKeys.spliterator(), false)
                    .filter(k -> !grandResult.containsKey(k))
                    .map(cacheContext::externalEntryKeyFor)
                    .collect(Collectors.toSet());

            if (missingExternalKeys.isEmpty()) {
                getLogger().trace("Cache {}: getBulk(): have all the requested entries cached", name);
                return grandResult;
            }
            getLogger().trace("Cache {}: getBulk(): not cached {} requested entries", name, missingExternalKeys.size());

            // Get the missing values.
            final Map<String, Optional<V>> candidateValues = directGetBulk(missingExternalKeys);

            return candidateValues.entrySet().stream().collect(
                    () -> grandResult,
                    (m, e) -> {
                        final Optional<V> result = e.getValue();
                        cacheContext.recordValue(cacheContext.internalEntryKeyFor(e.getKey()), result);
                        m.put(cacheContext.internalEntryKeyFor(e.getKey()), result);
                    },
                    Map::putAll
            );
        });
    }

    @Nonnull
    @Override
    public final CompletionStage<Map<String, V>> getBulk(
            Function<Set<String>, Map<String, V>> factory, Iterable<String> internalKeys) {
        return perform(() -> {
            if (isEmpty(internalKeys)) {
                return new HashMap<>();
            }

            // Get the recorded values first and collect the ones that have values.
            final AbstractExternalCacheRequestContext<V> cacheContext = ensureCacheContext();
            final Map<String, V> grandResult = checkValuesRecorded(internalKeys).entrySet().stream()
                    .filter(e -> e.getValue().isPresent())
                    .collect(Collectors.toMap(Map.Entry::getKey, e -> e.getValue().get()));

            // Calculate the candidate externalKeys for the entries that are missing
            final Set<String> candidateMissingExternalKeys = StreamSupport.stream(internalKeys.spliterator(), false)
                    .filter(k -> !grandResult.containsKey(k))
                    .map(cacheContext::externalEntryKeyFor)
                    .collect(Collectors.toSet());

            // Bail out if we have all the entries requested
            if (candidateMissingExternalKeys.isEmpty()) {
                getLogger().trace("Cache {}: getBulk(Function): had all the requested entries cached", name);
                return grandResult;
            }
            getLogger().trace("Cache {}: getBulk(Function): checking external cache for {} keys",
                    name, candidateMissingExternalKeys.size());

            final Map<String, V> missingValues = handleCreation(factory, candidateMissingExternalKeys);
            cacheContext.recordValues(missingValues);
            grandResult.putAll(missingValues);

            return grandResult;
        });
    }
}
