package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.PutPolicy;
import com.atlassian.vcache.TransactionalExternalCache;

import javax.annotation.Nonnull;

import static com.atlassian.vcache.internal.MetricLabel.TIMED_PUT_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_ALL_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link TransactionalExternalCache} that records metrics.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class TimedTransactionalExternalCache<V>
        extends TimedExternalCache<V>
        implements TransactionalExternalCache<V> {
    private final TransactionalExternalCache<V> delegate;

    TimedTransactionalExternalCache(MetricsCollector metricsCollector, TransactionalExternalCache<V> delegate) {
        super(metricsCollector);
        this.delegate = requireNonNull(delegate);
    }

    @Nonnull
    @Override
    protected TransactionalExternalCache<V> getDelegate() {
        return delegate;
    }

    @Override
    public void put(String key, V value, PutPolicy policy) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_PUT_CALL, t))) {
            getDelegate().put(key, value, policy);
        }
    }

    @Override
    public void remove(Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_REMOVE_CALL, t))) {
            getDelegate().remove(keys);
        }
    }

    @Override
    public void removeAll() {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_REMOVE_ALL_CALL, t))) {
            getDelegate().removeAll();
        }
    }
}
