package com.atlassian.vcache.internal.core.metrics;

import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a supplier used to create missing values. Calls the supplied handler with
 * the elapsed time taken in nanoseconds, and {@link Optional#empty()} means it was was not called.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
public class TimedSupplier<V> implements Supplier<V>, AutoCloseable {
    private final Supplier<? extends V> delegate;
    private final Consumer<Optional<Long>> handler;

    private Optional<Long> elapsedDuration = Optional.empty();

    public TimedSupplier(Supplier<V> delegate, Consumer<Optional<Long>> handler) {
        this.delegate = requireNonNull(delegate);
        this.handler = requireNonNull(handler);
    }

    @Override
    public V get() {
        try (ElapsedTimer ignored = new ElapsedTimer(t -> elapsedDuration = Optional.of(t))) {
            return delegate.get();
        }
    }

    @Override
    public void close() {
        handler.accept(elapsedDuration);
    }
}
