package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;

import javax.annotation.Nonnull;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_BYTES_MARSHALLED;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_BYTES_UNMARSHALLED;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_MARSHALL;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_UNMARSHALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_MARSHALL_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_UNMARSHALL_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link Marshaller} that records metrics.
 *
 * @param <T> the value type
 * @since 1.0.0
 */
class TimedMarshaller<T> implements Marshaller<T> {
    private final Marshaller<T> delegate;
    private final MetricsCollector metricsCollector;
    private final String cacheName;

    TimedMarshaller(Marshaller<T> delegate, MetricsCollector metricsCollector, String cacheName) {
        this.delegate = requireNonNull(delegate);
        this.metricsCollector = requireNonNull(metricsCollector);
        this.cacheName = requireNonNull(cacheName);
    }

    @Nonnull
    @Override
    public byte[] marshall(T obj) throws MarshallerException {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, cacheName, TIMED_MARSHALL_CALL, t))) {
            final byte[] result = delegate.marshall(obj);
            metricsCollector.record(EXTERNAL, cacheName, NUMBER_OF_BYTES_MARSHALLED, result.length);
            return result;
        } catch (MarshallerException me) {
            metricsCollector.record(EXTERNAL, cacheName, NUMBER_OF_FAILED_MARSHALL, 1);
            throw me;
        }
    }

    @Nonnull
    @Override
    public T unmarshall(byte[] raw) throws MarshallerException {
        metricsCollector.record(EXTERNAL, cacheName, NUMBER_OF_BYTES_UNMARSHALLED, raw.length);
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, cacheName, TIMED_UNMARSHALL_CALL, t))) {
            return delegate.unmarshall(raw);
        } catch (MarshallerException me) {
            metricsCollector.record(EXTERNAL, cacheName, NUMBER_OF_FAILED_UNMARSHALL, 1);
            throw me;
        }
    }
}
