package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.LocalCacheOperations;
import com.atlassian.vcache.internal.MetricLabel;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_HITS;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_MISSES;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_GET_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_PUT_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_ALL_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_SUPPLIER_CALL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link LocalCacheOperations} that records metrics.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0.0
 */
abstract class TimedLocalCacheOperations<K, V>
        implements LocalCacheOperations<K, V> {
    protected final String cacheName;
    protected final CacheType cacheType;
    protected final MetricsCollector metricsCollector;

    TimedLocalCacheOperations(String cacheName, CacheType cacheType, MetricsCollector metricsCollector) {
        this.cacheName = requireNonNull(cacheName);
        this.cacheType = requireNonNull(cacheType);
        this.metricsCollector = requireNonNull(metricsCollector);
    }

    @Nonnull
    protected abstract LocalCacheOperations<K, V> getDelegate();

    @Nonnull
    @Override
    public Optional<V> get(K key) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_GET_CALL, t))) {
            final Optional<V> result = getDelegate().get(key);
            metricsCollector.record(
                    cacheType,
                    cacheName,
                    result.isPresent() ? MetricLabel.NUMBER_OF_HITS : MetricLabel.NUMBER_OF_MISSES,
                    1);

            return result;
        }
    }

    @Nonnull
    @Override
    public V get(K key, Supplier<? extends V> supplier) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_GET_CALL, t));
             TimedSupplier<? extends V> timedSupplier = new TimedSupplier<>(supplier, this::handleTimedSupplier)) {
            return getDelegate().get(key, timedSupplier);
        }
    }

    @Override
    public void put(K key, V value) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_PUT_CALL, t))) {
            getDelegate().put(key, value);
        }
    }

    @Nonnull
    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_PUT_CALL, t))) {
            return getDelegate().putIfAbsent(key, value);
        }
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_PUT_CALL, t))) {
            return getDelegate().replaceIf(key, currentValue, newValue);
        }
    }

    @Override
    public boolean removeIf(K key, V value) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_REMOVE_CALL, t))) {
            return getDelegate().removeIf(key, value);
        }
    }

    @Override
    public void remove(K key) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_REMOVE_CALL, t))) {
            getDelegate().remove(key);
        }
    }

    @Override
    public void removeAll() {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(cacheType, cacheName, TIMED_REMOVE_ALL_CALL, t))) {
            getDelegate().removeAll();
        }
    }

    private void handleTimedSupplier(Optional<Long> time) {
        if (time.isPresent()) {
            metricsCollector.record(cacheType, cacheName, TIMED_SUPPLIER_CALL, time.get());
        }
        metricsCollector.record(
                cacheType,
                cacheName,
                time.isPresent() ? NUMBER_OF_MISSES : NUMBER_OF_HITS,
                1);
    }
}
