package com.atlassian.vcache.internal.core.metrics;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Function;

import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a function used to create missing values. Calls the supplied handler with
 * the elapsed time taken in nanoseconds, and {@link Optional#empty()} means it was was not called.
 * All passes the number of keys passed to the factory.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0.0
 */
public class TimedFactory<K, V> implements Function<Set<K>, Map<K, V>>, AutoCloseable {
    private final Function<Set<K>, Map<K, V>> delegate;
    private final BiConsumer<Optional<Long>, Long> handler;

    private Optional<Long> elapsedDuration = Optional.empty();
    private long numberOfKeys;

    public TimedFactory(
            Function<Set<K>, Map<K, V>> delegate, BiConsumer<Optional<Long>, Long> handler) {
        this.delegate = requireNonNull(delegate);
        this.handler = requireNonNull(handler);
    }

    @Override
    public Map<K, V> apply(Set<K> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(t -> {
            elapsedDuration = Optional.of(t);
            numberOfKeys = (long) keys.size();
        })) {
            return delegate.apply(keys);
        }
    }

    @Override
    public void close() {
        handler.accept(elapsedDuration, numberOfKeys);
    }

    public long getNumberOfKeys() {
        return numberOfKeys;
    }
}
