package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.CasIdentifier;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalWriteOperationsUnbuffered;
import com.atlassian.vcache.IdentifiedValue;

import javax.annotation.Nonnull;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletionStage;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_IDENTIFIED_GET;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_IDENTIFIED_REMOVE;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_IDENTIFIED_REPLACE;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_HITS;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_MISSES;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_IDENTIFIED_GET_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_IDENTIFIED_REMOVE_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_IDENTIFIED_REPLACE_CALL;
import static com.atlassian.vcache.internal.core.VCacheCoreUtils.whenPositive;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static com.atlassian.vcache.internal.core.metrics.TimedUtils.whenCompletableFuture;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link DirectExternalCache} that records metrics.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class TimedDirectExternalCache<V>
        extends TimedExternalWriteOperationsUnbuffered<V>
        implements DirectExternalCache<V> {
    private final DirectExternalCache<V> delegate;

    TimedDirectExternalCache(MetricsCollector metricsCollector, DirectExternalCache<V> delegate) {
        super(metricsCollector);
        this.delegate = requireNonNull(delegate);
    }

    @Nonnull
    @Override
    protected ExternalWriteOperationsUnbuffered<V> getDelegateOps() {
        return delegate;
    }

    @Nonnull
    @Override
    protected DirectExternalCache<V> getDelegate() {
        return delegate;
    }

    @Nonnull
    @Override
    public CompletionStage<Optional<IdentifiedValue<V>>> getIdentified(String key) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_IDENTIFIED_GET_CALL, t))) {
            final CompletionStage<Optional<IdentifiedValue<V>>> result = getDelegate().getIdentified(key);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_FAILED_IDENTIFIED_GET, 1);
                } else {
                    final Optional<IdentifiedValue<V>> rj = future.join();
                    metricsCollector.record(
                            EXTERNAL,
                            getDelegate().getName(),
                            rj.isPresent() ? NUMBER_OF_HITS : NUMBER_OF_MISSES,
                            1);
                }
            });

            return result;
        }
    }

    @Nonnull
    @Override
    public CompletionStage<Map<String, Optional<IdentifiedValue<V>>>> getBulkIdentified(Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_IDENTIFIED_GET_CALL, t))) {
            final CompletionStage<Map<String, Optional<IdentifiedValue<V>>>> result = getDelegate().getBulkIdentified(keys);

            whenCompletableFuture(result, future -> {

                if (future.isCompletedExceptionally()) {
                    metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_FAILED_IDENTIFIED_GET, 1);
                } else {
                    final Map<String, Optional<IdentifiedValue<V>>> rj = future.join();
                    final long hits = rj.values().stream().filter(Optional::isPresent).count();

                    whenPositive(hits,
                            v -> metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_HITS, v));
                    whenPositive(rj.size() - hits,
                            v -> metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_MISSES, v));
                }
            });

            return result;
        }
    }

    @Nonnull
    @Override
    public CompletionStage<Boolean> removeIf(String key, CasIdentifier casId) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_IDENTIFIED_REMOVE_CALL, t))) {
            final CompletionStage<Boolean> result = getDelegate().removeIf(key, casId);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_FAILED_IDENTIFIED_REMOVE, 1);
                }
            });

            return result;
        }
    }

    @Nonnull
    @Override
    public CompletionStage<Boolean> replaceIf(String key, CasIdentifier casId, V newValue) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsCollector.record(EXTERNAL, getDelegate().getName(), TIMED_IDENTIFIED_REPLACE_CALL, t))) {
            final CompletionStage<Boolean> result = getDelegate().replaceIf(key, casId, newValue);
            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsCollector.record(EXTERNAL, getDelegate().getName(), NUMBER_OF_FAILED_IDENTIFIED_REPLACE, 1);
                }
            });

            return result;
        }
    }
}
