package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.RequestCache;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.internal.MetricLabel;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.RequestMetrics;
import com.atlassian.vcache.internal.core.TransactionControl;

import javax.annotation.Nonnull;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link MetricsCollector}.
 *
 * @since 1.0.0
 */
public class DefaultMetricsCollector implements MetricsCollector {

    private final Supplier<RequestContext> contextSupplier;

    public DefaultMetricsCollector(Supplier<RequestContext> contextSupplier) {
        this.contextSupplier = requireNonNull(contextSupplier);
    }

    @Override
    public void record(CacheType cacheType, String cacheName, MetricLabel metricLabel, long sample) {
        obtainMetrics(contextSupplier.get()).record(cacheName, cacheType, metricLabel, sample);
    }

    @Nonnull
    @Override
    public RequestMetrics obtainRequestMetrics(RequestContext context) {
        return obtainMetrics(context);
    }

    @Nonnull
    @Override
    public TransactionControl wrap(TransactionControl control, String cacheName) {
        return new TimedTransactionControl(control, this, cacheName);
    }

    @Nonnull
    @Override
    public <T> Marshaller<T> wrap(Marshaller<T> marshaller, String cacheName) {
        return new TimedMarshaller<>(marshaller, this, cacheName);
    }

    @Nonnull
    @Override
    public <K, V> JvmCache<K, V> wrap(JvmCache<K, V> cache) {
        return new TimedJvmCache<>(cache, this);
    }

    @Nonnull
    @Override
    public <K, V> RequestCache<K, V> wrap(RequestCache<K, V> cache) {
        return new TimedRequestCache<>(cache, this);
    }

    @Nonnull
    @Override
    public <V> DirectExternalCache<V> wrap(DirectExternalCache<V> cache) {
        return new TimedDirectExternalCache<>(this, cache);
    }

    @Nonnull
    @Override
    public <V> StableReadExternalCache<V> wrap(StableReadExternalCache<V> cache) {
        return new TimedStableReadExternalCache<>(this, cache);
    }

    @Nonnull
    @Override
    public <V> TransactionalExternalCache<V> wrap(TransactionalExternalCache<V> cache) {
        return new TimedTransactionalExternalCache<>(this, cache);
    }

    @Nonnull
    MutableRequestMetrics obtainMetrics(RequestContext context) {
        return context.computeIfAbsent(this, DefaultRequestMetrics::new);
    }
}
