package com.atlassian.vcache.internal.core.cas;

import com.atlassian.vcache.CasIdentifier;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.IdentifiedValue;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;
import com.atlassian.vcache.internal.core.DefaultIdentifiedValue;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.io.Serializable;
import java.util.Optional;

import static com.atlassian.vcache.ExternalCacheException.Reason.MARSHALLER_FAILURE;
import static com.atlassian.vcache.ExternalCacheException.Reason.UNCLASSIFIED_FAILURE;

/**
 * Contains common methods for dealing with {@link IdentifiedData}.
 *
 * @since 1.1.0
 */
public class IdentifiedUtils {
    private static final Logger log = LoggerFactory.getLogger(IdentifiedUtils.class);

    @Nonnull
    public static <V> IdentifiedData marshall(V data, Optional<Marshaller<V>> valueMarshaller) throws ExternalCacheException {
        try {
            return valueMarshaller.isPresent()
                    ? new IdentifiedDataBytes(valueMarshaller.get().marshall(data))
                    : new IdentifiedDataSerializable((Serializable) data);
        } catch (MarshallerException e) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, e);
        }
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    public static <V> Optional<V> unmarshall(@Nullable IdentifiedData idata, Optional<Marshaller<V>> valueMarshaller)
            throws ExternalCacheException {
        if (idata == null) {
            return Optional.empty();
        }

        try {
            return valueMarshaller.isPresent()
                    ? Optional.of(valueMarshaller.get().unmarshall(((IdentifiedDataBytes) idata).getBytes()))
                    : Optional.of((V) ((IdentifiedDataSerializable) idata).getObject());
        } catch (MarshallerException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    public static <V> Optional<IdentifiedValue<V>> unmarshallIdentified(
            @Nullable IdentifiedData idata, Optional<Marshaller<V>> valueMarshaller) {
        if (idata == null) {
            return Optional.empty();
        }

        try {
            final V value = valueMarshaller.isPresent()
                    ? valueMarshaller.get().unmarshall(((IdentifiedDataBytes) idata).getBytes())
                    : (V) ((IdentifiedDataSerializable) idata).getObject();
            return Optional.of(new DefaultIdentifiedValue<>(idata, value));
        } catch (MarshallerException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    @Nonnull
    public static IdentifiedData safeCast(CasIdentifier casId) {
        if (casId instanceof IdentifiedData) {
            return (IdentifiedData) casId;
        }

        log.warn("Passed an unknown CasIdentifier instance of class {}.", casId.getClass().getName());
        throw new ExternalCacheException(UNCLASSIFIED_FAILURE);
    }
}
