package com.atlassian.vcache.internal.core;

import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.LongConsumer;

import static com.atlassian.vcache.ExternalCacheException.Reason.MARSHALLER_FAILURE;

/**
 * Contains common implementation methods.
 *
 * @since 1.0.0
 */
public class VCacheCoreUtils {
    /**
     * Returns a successfully completed {@link CompletionStage} with the specified result.
     *
     * @param value the result for the future
     * @param <T>   the type of the result
     * @return a successfully completed {@link CompletionStage} with the specified result.
     */
    @Nonnull
    public static <T> CompletionStage<T> successful(T value) {
        final CompletableFuture<T> result = new CompletableFuture<>();
        result.complete(value);
        return result;
    }

    /**
     * Returns a failed {@link CompletionStage} with the specified cause.
     *
     * @param cause the cause of the failure
     * @param <T>   the type of the result
     * @return a failed {@link CompletionStage} with the specified cause.
     */
    @Nonnull
    public static <T> CompletionStage<T> failed(Throwable cause) {
        final CompletableFuture<T> result = new CompletableFuture<>();
        result.completeExceptionally(cause);
        return result;
    }

    public static int roundUpToSeconds(Duration time) {
        if (time.isNegative()) {
            throw new IllegalArgumentException("duration cannot be negative: " + time);
        }

        final long result = (time.getNano() > 0) ? time.getSeconds() + 1 : time.getSeconds();

        if (result > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("duration exceeds maximum number that can be held in an int");
        }

        return (int) result;
    }

    public static void whenPositive(long number, LongConsumer handler) {
        if (number > 0) {
            handler.accept(number);
        }
    }

    @Nonnull
    public static <V> byte[] marshall(V data, Marshaller<V> valueMarshaller) throws ExternalCacheException {
        try {
            return valueMarshaller.marshall(data);
        } catch (MarshallerException e) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, e);
        }
    }

    @Nonnull
    public static <V> Optional<V> unmarshall(@Nullable byte[] data, Marshaller<V> valueMarshaller) throws ExternalCacheException {
        if (data == null) {
            return Optional.empty();
        }

        try {
            return Optional.of(valueMarshaller.unmarshall(data));
        } catch (MarshallerException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    public static boolean isEmpty(Iterable<String> iter) {
        return !iter.iterator().hasNext();
    }
}
