package com.atlassian.vcache.internal.core;

import com.atlassian.vcache.ChangeRate;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.ExternalCacheSettingsBuilder;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.JvmCacheSettingsBuilder;
import com.atlassian.vcache.VCacheException;
import com.atlassian.vcache.internal.ExternalCacheDetails;
import com.atlassian.vcache.internal.JvmCacheDetails;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.time.Duration;
import java.util.Optional;

import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link VCacheCreationHandler} that enforces maximum allowed values.
 *
 * @since 1.0.0
 */
public class DefaultVCacheCreationHandler implements VCacheCreationHandler {
    private static final Logger log = LoggerFactory.getLogger(DefaultVCacheCreationHandler.class);

    private final int maxEntries;
    private final Duration maxDefaultTtl;
    private final int maxEntryCountHint;
    private final ChangeRate defaultDataChangeRateHint;
    private final ChangeRate defaultEntryGrowthRateHint;


    public DefaultVCacheCreationHandler(
            int maxEntries,
            Duration maxDefaultTtl,
            int maxEntryCountHint,
            ChangeRate defaultDataChangeRateHint,
            ChangeRate defaultEntryGrowthRateHint) {
        this.maxEntries = maxEntries;
        this.maxDefaultTtl = requireNonNull(maxDefaultTtl);
        this.maxEntryCountHint = maxEntryCountHint;
        this.defaultDataChangeRateHint = requireNonNull(defaultDataChangeRateHint);
        this.defaultEntryGrowthRateHint = requireNonNull(defaultEntryGrowthRateHint);
    }

    @Nonnull
    @Override
    public JvmCacheSettings jvmCacheCreation(JvmCacheDetails details) throws VCacheException {
        final JvmCacheSettings candidateSettings = details.getSettings();
        final JvmCacheSettingsBuilder bob = new JvmCacheSettingsBuilder(candidateSettings);

        enforceInteger(candidateSettings.getMaxEntries(), maxEntryCountHint, () -> {
            log.trace("Cache {}: forcing maxEntries to be {}", details.getName(), maxEntries);
            bob.maxEntries(maxEntries);
        });

        enforceDuration(candidateSettings.getDefaultTtl(), maxDefaultTtl, () -> {
            log.trace("Cache {}: forcing defaultTtl to be {}", details.getName(), maxDefaultTtl);
            bob.defaultTtl(maxDefaultTtl);
        });

        return bob.build();
    }

    @Override
    public void requestCacheCreation(String name) {
        // Just allow it
    }

    @Nonnull
    @Override
    public ExternalCacheSettings externalCacheCreation(ExternalCacheDetails details) {
        final ExternalCacheSettings candidateSettings = details.getSettings();
        final ExternalCacheSettingsBuilder bob = new ExternalCacheSettingsBuilder(candidateSettings);

        enforceDuration(candidateSettings.getDefaultTtl(), maxDefaultTtl, () -> {
            log.trace("Cache {}: forcing defaultTtl to be {}", details.getName(), maxDefaultTtl);
            bob.defaultTtl(maxDefaultTtl);
        });

        enforceInteger(candidateSettings.getEntryCountHint(), maxEntryCountHint, () -> {
            log.trace("Cache {}: forcing entryCountHint to be {}", details.getName(), maxEntryCountHint);
            bob.entryCountHint(maxEntryCountHint);
        });

        enforceRate(candidateSettings.getDataChangeRateHint(), () -> {
            log.trace("Cache {}: forcing dataChangeRateHint to be {}", details.getName(), defaultDataChangeRateHint);
            bob.dataChangeRateHint(defaultDataChangeRateHint);
        });
        enforceRate(candidateSettings.getEntryGrowthRateHint(), () -> {
            log.trace("Cache {}: forcing entryGrowthRateHint to be {}", details.getName(), defaultEntryGrowthRateHint);
            bob.entryGrowthRateHint(defaultEntryGrowthRateHint);
        });

        return bob.build();
    }

    private void enforceDuration(Optional<Duration> opt, Duration max, Runnable handler) {
        if (!opt.isPresent() || (opt.get().compareTo(max) > 0)) {
            handler.run();
        }
    }

    private void enforceInteger(Optional<Integer> opt, int max, Runnable handler) {
        if (!opt.isPresent() || (opt.get() > max)) {
            handler.run();
        }
    }

    private void enforceRate(Optional<ChangeRate> opt, Runnable handler) {
        if (!opt.isPresent()) {
            handler.run();
        }
    }
}
