package com.atlassian.vcache.internal.core;

import com.atlassian.vcache.internal.RequestContext;

import javax.annotation.Nonnull;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * An implementation of the {@link RequestContext} that is <b>NOT</b> thread-safe.
 *
 * @since 1.0
 */
public class DefaultRequestContext implements RequestContext {
    private final String partitionId;
    private final Map<Object, Object> map = new HashMap<>();

    public DefaultRequestContext(String partitionId) {
        this.partitionId = requireNonNull(partitionId);
    }

    @Nonnull
    @Override
    public String partitionIdentifier() {
        return partitionId;
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    @Override
    public <T> T computeIfAbsent(Object key, Supplier<T> supplier) {
        // Written this way due to the JDK bug - https://bugs.openjdk.java.net/browse/JDK-8071667.
        //     return (T) map.computeIfAbsent(requireNonNull(key), o -> requireNonNull(supplier.get()));

        if (map.containsKey(requireNonNull(key))) {
            return (T) map.get(key);
        } else {
            final T v = requireNonNull(supplier.get());
            map.put(key, v);
            return v;
        }
    }

    @SuppressWarnings("unchecked")
    @Nonnull
    @Override
    public <T> Optional<T> get(Object key) {
        return Optional.ofNullable((T) map.get(key));
    }
}
