package com.atlassian.vcache.internal.core;

import com.atlassian.vcache.RequestCache;
import com.atlassian.vcache.internal.RequestContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nonnull;
import java.util.HashMap;
import java.util.Optional;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Implementation of {@link RequestCache} that uses a delegate {@link HashMap}
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
public class DefaultRequestCache<K, V> implements RequestCache<K, V> {
    private static final Logger log = LoggerFactory.getLogger(DefaultRequestCache.class);

    private final String name;
    private final Supplier<RequestContext> contextSupplier;

    public DefaultRequestCache(String name, Supplier<RequestContext> contextSupplier) {
        this.name = requireValidCacheName(name);
        this.contextSupplier = requireNonNull(contextSupplier);
    }

    @Nonnull
    @Override
    public Optional<V> get(K key) {
        return Optional.ofNullable(ensureDelegate().get(key));
    }

    @Nonnull
    @Override
    public V get(K key, Supplier<? extends V> supplier) {
        return ensureDelegate().computeIfAbsent(requireNonNull(key), k -> requireNonNull(supplier.get()));
    }

    @Override
    public void put(K key, V value) {
        ensureDelegate().put(requireNonNull(key), requireNonNull(value));
    }

    @Nonnull
    @Override
    public Optional<V> putIfAbsent(K key, V value) {
        return Optional.ofNullable(
                ensureDelegate().putIfAbsent(requireNonNull(key), requireNonNull(value)));
    }

    @Override
    public boolean replaceIf(K key, V currentValue, V newValue) {
        return ensureDelegate().replace(requireNonNull(key), requireNonNull(currentValue), requireNonNull(newValue));
    }

    @Override
    public boolean removeIf(K key, V value) {
        return ensureDelegate().remove(requireNonNull(key), requireNonNull(value));
    }

    @Override
    public void remove(K key) {
        ensureDelegate().remove(key);
    }

    @Override
    public void removeAll() {
        ensureDelegate().clear();
    }

    @Nonnull
    @Override
    public String getName() {
        return name;
    }

    private HashMap<K, V> ensureDelegate() {
        final RequestContext requestContext = contextSupplier.get();
        return requestContext.computeIfAbsent(this, HashMap::new);
    }
}
