package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.VCacheException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;

import static java.util.Objects.requireNonNull;

/**
 * A lock for caches. It provides the common logic, such as timeouts for acquiring locks.
 * @since 1.9.0
 */
public class VCacheLock {
    private static final Logger log = LoggerFactory.getLogger(VCacheLock.class);

    private final ReentrantLock lock = new ReentrantLock();
    private final String cacheName;
    private final long lockTimeoutMillis;

    public VCacheLock(String cacheName, Duration lockTimeout) {
        this.cacheName = requireNonNull(cacheName);
        this.lockTimeoutMillis = lockTimeout.toMillis();
    }

    public <R> R withLock(Supplier<R> supplier) {
        lockWithTimeout();
        try {
            return supplier.get();
        } finally {
            lock.unlock();
        }
    }

    public void withLock(Runnable runner) {
        lockWithTimeout();
        try {
            runner.run();
        } finally {
            lock.unlock();
        }
    }

    private void lockWithTimeout() {
        try {
            if (!lock.tryLock(lockTimeoutMillis, TimeUnit.MILLISECONDS)) {
                log.warn("Timed out waiting for lock on cache: {}", cacheName);
                throw new VCacheException("Timed out waiting for lock on cache: " + cacheName);
            }
        } catch (InterruptedException e) {
            Thread.interrupted();
            log.warn("Interrupted whilst waiting for a lock on cache: ", cacheName, e);
            throw new VCacheException("Interrupted waiting for lock on cache: " + cacheName, e);
        }
    }
}
