package com.atlassian.vcache.internal.core.service;

import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.ChangeRate;
import com.atlassian.vcache.DirectExternalCache;
import com.atlassian.vcache.ExternalCache;
import com.atlassian.vcache.ExternalCacheException;
import com.atlassian.vcache.ExternalCacheSettings;
import com.atlassian.vcache.JvmCache;
import com.atlassian.vcache.JvmCacheSettings;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.RequestCache;
import com.atlassian.vcache.RequestCacheSettings;
import com.atlassian.vcache.StableReadExternalCache;
import com.atlassian.vcache.TransactionalExternalCache;
import com.atlassian.vcache.VCacheFactory;
import com.atlassian.vcache.internal.BegunTransactionalActivityHandler;
import com.atlassian.vcache.internal.ExternalCacheDetails;
import com.atlassian.vcache.internal.JvmCacheDetails;
import com.atlassian.vcache.internal.RequestCacheDetails;
import com.atlassian.vcache.internal.RequestContext;
import com.atlassian.vcache.internal.RequestMetrics;
import com.atlassian.vcache.internal.VCacheCreationHandler;
import com.atlassian.vcache.internal.VCacheLifecycleManager;
import com.atlassian.vcache.internal.VCacheManagement;
import com.atlassian.vcache.internal.VCacheSettingsDefaultsProvider;
import com.atlassian.vcache.internal.core.DefaultExternalCacheDetails;
import com.atlassian.vcache.internal.core.DefaultJvmCacheDetails;
import com.atlassian.vcache.internal.core.DefaultRequestCacheDetails;
import com.atlassian.vcache.internal.core.DefaultTransactionControlManager;
import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import com.atlassian.vcache.internal.core.TransactionControlManager;
import com.atlassian.vcache.internal.core.metrics.MetricsCollector;
import com.google.common.annotations.VisibleForTesting;
import org.slf4j.Logger;

import java.time.Duration;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.NameValidator.requireValidCacheName;
import static java.util.Objects.requireNonNull;

/**
 * Base implementation of a service.
 *
 * @since 1.0.0
 */
public abstract class AbstractVCacheService implements VCacheFactory, VCacheManagement, VCacheLifecycleManager {
    // List of Marshaller class names that support Serializable values. This is a slightly
    // dodgy approach, but the rationale is to not be forced to make implementations
    // visible, or to make it an attribute of a Marshaller. The fact is, this is a temporary
    // hack for internal use only.
    private static final Set<String> SERIALIZABLE_MARSHALLER_CLASS_NAMES =
            Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
                    "com.atlassian.vcache.marshallers.StringMarshaller",
                    "com.atlassian.vcache.marshallers.JavaSerializationMarshaller")));
    private static final Set<String> SERIALIZABLE_MARSHALLING_CLASS_NAMES =
            Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
                    "com.atlassian.marshalling.jdk.JavaSerializationMarshalling",
                    "com.atlassian.marshalling.jdk.StringMarshalling")));

    protected final Supplier<RequestContext> workContextContextSupplier;
    protected final Supplier<RequestContext> threadLocalContextSupplier;
    protected final TransactionControlManager transactionControlManager;
    protected final MetricsCollector metricsCollector;

    protected final ExternalCacheKeyGenerator externalCacheKeyGenerator;
    protected final Duration lockTimeout;
    private final VCacheSettingsDefaultsProvider defaultsProvider;
    private final VCacheCreationHandler creationHandler;

    private final ConcurrentHashMap<String, JvmCache> jvmCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, JvmCacheDetails> jvmCacheDetailsMap = new ConcurrentHashMap<>();

    private final ConcurrentHashMap<String, RequestCache> requestCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, RequestCacheDetails> requestCacheDetailsMap = new ConcurrentHashMap<>();

    private final ConcurrentHashMap<String, ExternalCache> externalCacheInstancesMap = new ConcurrentHashMap<>();
    private final ConcurrentHashMap<String, ExternalCacheDetails> externalCacheDetailsMap = new ConcurrentHashMap<>();

    public AbstractVCacheService(Supplier<RequestContext> threadLocalContextSupplier,
                                 Supplier<RequestContext> workContextContextSupplier,
                                 VCacheSettingsDefaultsProvider defaultsProvider,
                                 VCacheCreationHandler creationHandler,
                                 MetricsCollector metricsCollector,
                                 ExternalCacheKeyGenerator externalCacheKeyGenerator,
                                 BegunTransactionalActivityHandler begunTransactionalActivityHandler,
                                 Duration lockTimeout) {
        this.threadLocalContextSupplier = requireNonNull(threadLocalContextSupplier);
        this.workContextContextSupplier = requireNonNull(workContextContextSupplier);
        this.defaultsProvider = requireNonNull(defaultsProvider);
        this.creationHandler = requireNonNull(creationHandler);
        this.metricsCollector = requireNonNull(metricsCollector);
        this.transactionControlManager = new DefaultTransactionControlManager(metricsCollector, begunTransactionalActivityHandler);
        this.externalCacheKeyGenerator = requireNonNull(externalCacheKeyGenerator);
        this.lockTimeout = requireNonNull(lockTimeout);
    }

    /**
     * Returns the logger for the implementation service.
     *
     * @return the logger for the implementation service.
     */
    protected abstract Logger log();

    /**
     * Creates a {@link JvmCache} with the supplied parameters.
     *
     * @param name     the name of the cache
     * @param settings the setting for the cache
     * @param <K>      the key type
     * @param <V>      the value type
     * @return a freshly minted {@link JvmCache}
     */
    protected abstract <K, V> JvmCache<K, V> createJvmCache(String name, JvmCacheSettings settings);

    /**
     * Creates a {@link TransactionalExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshalling  the marshalling pair for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link TransactionalExternalCache}
     */
    protected abstract <V> TransactionalExternalCache<V> createTransactionalExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable);

    /**
     * Creates a {@link StableReadExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshalling  the marshalling pair for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link StableReadExternalCache}
     */
    protected abstract <V> StableReadExternalCache<V> createStableReadExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable);

    /**
     * Creates a {@link DirectExternalCache} with the supplied parameters.
     *
     * @param name              the name of the cache
     * @param settings          the settings for the cache
     * @param valueMarshalling  the marshalling pair for values
     * @param valueSerializable whether the values are {@link java.io.Serializable}
     * @param <V>               the value type
     * @return a freshly minted {@link DirectExternalCache}
     */
    protected abstract <V> DirectExternalCache<V> createDirectExternalCache(
            String name, ExternalCacheSettings settings, MarshallingPair<V> valueMarshalling, boolean valueSerializable);

    @Override
    @SuppressWarnings("unchecked")
    public <K, V> JvmCache<K, V> getJvmCache(String name, JvmCacheSettings settings) {
        return jvmCacheInstancesMap.computeIfAbsent(requireValidCacheName(name), s -> {
            log().trace("Cache {}: creating the instance", name);
            final JvmCacheSettings candidateSettings = defaultsProvider.getJvmDefaults(name).override(settings);
            final JvmCacheSettings finalSettings =
                    creationHandler.jvmCacheCreation(new DefaultJvmCacheDetails(name, candidateSettings));
            jvmCacheDetailsMap.put(name, new DefaultJvmCacheDetails(name, finalSettings));

            return metricsCollector.wrap(createJvmCache(name, finalSettings));
        });
    }

    @SuppressWarnings("unchecked")
    public <K, V> RequestCache<K, V> getRequestCache(String name, RequestCacheSettings settings) {
        return requestCacheInstancesMap.computeIfAbsent(requireValidCacheName(name), s -> {
            log().trace("DefaultRequestCache {}: creating the instance", name);
            creationHandler.requestCacheCreation(name);
            requestCacheDetailsMap.put(name, new DefaultRequestCacheDetails(name));
            if (settings.getChangeRate().equals(ChangeRate.NONE)) {
                return metricsCollector.wrap(new ReadOptimisedRequestCache(name, workContextContextSupplier, lockTimeout));
            } else {
                return metricsCollector.wrap(new DefaultRequestCache(name, workContextContextSupplier, lockTimeout));
            }
        });
    }

    @Override
    public <V> TransactionalExternalCache<V> getTransactionalExternalCache(String name, MarshallingPair<V> valueMarshalling, ExternalCacheSettings settings) {
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(valueMarshalling, name);

        final TransactionalExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.FULLY, settings, finalSettings ->
                        createTransactionalExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshalling)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public <V> TransactionalExternalCache<V> getTransactionalExternalCache(
            String name, @SuppressWarnings("deprecation") Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final MarshallingPair<V> marshallingPair =
                new MarshallingPair<>(valueMarshaller, valueMarshaller);
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(marshallingPair, name);

        final TransactionalExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.FULLY, settings, finalSettings ->
                        createTransactionalExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public <V> StableReadExternalCache<V> getStableReadExternalCache(String name, MarshallingPair<V> valueMarshalling, ExternalCacheSettings settings) {
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(valueMarshalling, name);
        final StableReadExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.READ_ONLY, settings, finalSettings ->
                        createStableReadExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshalling)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public <V> StableReadExternalCache<V> getStableReadExternalCache(
            String name, @SuppressWarnings("deprecation") Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final MarshallingPair<V> marshallingPair =
                new MarshallingPair<>(valueMarshaller, valueMarshaller);
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(marshallingPair, name);

        final StableReadExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.READ_ONLY, settings, finalSettings ->
                        createStableReadExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public <V> DirectExternalCache<V> getDirectExternalCache(String name, MarshallingPair<V> valueMarshalling, ExternalCacheSettings settings) {
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(valueMarshalling, name);
        final DirectExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.NEVER, settings, finalSettings ->
                        createDirectExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshalling)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public <V> DirectExternalCache<V> getDirectExternalCache(
            String name, @SuppressWarnings("deprecation") Marshaller<V> valueMarshaller, ExternalCacheSettings settings) {
        final MarshallingPair<V> marshallingPair =
                new MarshallingPair<>(valueMarshaller, valueMarshaller);
        final MarshallingPair<V> wrappedMarshalling = metricsCollector.wrap(marshallingPair, name);

        final DirectExternalCache<V> cache = obtainCache(
                name, ExternalCacheDetails.BufferPolicy.NEVER, settings, finalSettings ->
                        createDirectExternalCache(
                                name, finalSettings, wrappedMarshalling, isValueSerializable(valueMarshaller)));
        return metricsCollector.wrap(cache);
    }

    @Override
    public void transactionSync(RequestContext context) {
        transactionControlManager.syncAll(context);
    }

    @Override
    public Set<String> transactionDiscard(RequestContext context) {
        return transactionControlManager.discardAll(context);
    }

    @Override
    public RequestMetrics metrics(RequestContext context) {
        return metricsCollector.obtainRequestMetrics(context);
    }

    @Override
    public Map<String, JvmCacheDetails> allJvmCacheDetails() {
        final Map<String, JvmCacheDetails> result = new HashMap<>();
        result.putAll(jvmCacheDetailsMap);
        return result;
    }

    @Override
    public Map<String, RequestCacheDetails> allRequestCacheDetails() {
        final Map<String, RequestCacheDetails> result = new HashMap<>();
        result.putAll(requestCacheDetailsMap);
        return result;
    }

    @Override
    public Map<String, ExternalCacheDetails> allExternalCacheDetails() {
        final Map<String, ExternalCacheDetails> result = new HashMap<>();
        result.putAll(externalCacheDetailsMap);
        return result;
    }

    @VisibleForTesting
    static <V> boolean isValueSerializable(@SuppressWarnings("deprecation") Marshaller<V> valueMarshaller) {
        return SERIALIZABLE_MARSHALLER_CLASS_NAMES.contains(valueMarshaller.getClass().getName());
    }

    @VisibleForTesting
    static <V> boolean isValueSerializable(MarshallingPair<V> valueMarshalling) {
        return SERIALIZABLE_MARSHALLING_CLASS_NAMES.contains(valueMarshalling.getMarshaller().getClass().getName())
                && SERIALIZABLE_MARSHALLING_CLASS_NAMES.contains(valueMarshalling.getUnmarshaller().getClass().getName());
    }

    @SuppressWarnings("unchecked")
    private <C extends ExternalCache> C obtainCache(
            String name,
            ExternalCacheDetails.BufferPolicy policy,
            ExternalCacheSettings settings,
            Function<ExternalCacheSettings, C> factory) {
        final ExternalCache candidateCache = externalCacheInstancesMap.compute(
                requireValidCacheName(name),
                (key, existing) -> {
                    // Check if attempting to change the policy, a no no
                    if ((existing != null) && (externalCacheDetailsMap.get(name).getPolicy() != policy)) {
                        log().warn("Cache {}: unable to create cache with policy {}, as one already configured with policy {}",
                                name, ExternalCacheDetails.BufferPolicy.READ_ONLY, externalCacheDetailsMap.get(name).getPolicy());
                        throw new ExternalCacheException(ExternalCacheException.Reason.CREATION_FAILURE);
                    }

                    // Now always create a new cache wrapper, regardless of whether one existed before.
                    log().trace("Cache {}: creating the instance with policy {}", name, policy);
                    final ExternalCacheSettings candidateSettings =
                            defaultsProvider.getExternalDefaults(name).override(settings);
                    final ExternalCacheSettings finalSettings = creationHandler.externalCacheCreation(
                            new DefaultExternalCacheDetails(name, policy, candidateSettings));
                    externalCacheDetailsMap.put(
                            name, new DefaultExternalCacheDetails(name, policy, finalSettings));
                    return factory.apply(finalSettings);
                });

        return (C) candidateCache;
    }
}
