package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.ExternalCache;

import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FACTORY_KEYS;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_GET;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_HITS;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_MISSES;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_FACTORY_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_GET_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_SUPPLIER_CALL;
import static com.atlassian.vcache.internal.core.VCacheCoreUtils.whenPositive;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static com.atlassian.vcache.internal.core.metrics.TimedUtils.whenCompletableFuture;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link ExternalCache} that records metrics.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
abstract class TimedExternalCache<V> implements ExternalCache<V> {
    protected final MetricsRecorder metricsRecorder;

    TimedExternalCache(MetricsRecorder metricsRecorder) {
        this.metricsRecorder = requireNonNull(metricsRecorder);
    }

    protected abstract ExternalCache<V> getDelegate();

    @Override
    public CompletionStage<Optional<V>> get(String key) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_GET_CALL, t))) {
            final CompletionStage<Optional<V>> result = getDelegate().get(key);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_GET, 1);
                } else {
                    final Optional<V> rj = future.join();
                    metricsRecorder.record(
                            getDelegate().getName(),
                            EXTERNAL,
                            rj.isPresent() ? NUMBER_OF_HITS : NUMBER_OF_MISSES,
                            1);
                }
            });

            return result;
        }
    }

    @Override
    public CompletionStage<V> get(String key, Supplier<V> supplier) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_GET_CALL, t));
             TimedSupplier<V> timedSupplier = new TimedSupplier<>(supplier, this::handleTimedSupplier)) {
            final CompletionStage<V> result = getDelegate().get(key, timedSupplier);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_GET, 1);
                } else {
                    metricsRecorder.record(
                            getDelegate().getName(),
                            EXTERNAL,
                            timedSupplier.wasInvoked() ? NUMBER_OF_MISSES : NUMBER_OF_HITS,
                            1);
                }
            });

            return result;
        }
    }

    @Override
    public CompletionStage<Map<String, Optional<V>>> getBulk(Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_GET_CALL, t))) {
            final CompletionStage<Map<String, Optional<V>>> result = getDelegate().getBulk(keys);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_GET, 1);
                } else {
                    final Map<String, Optional<V>> rj = future.join();

                    final long hits = rj.values().stream().filter(Optional::isPresent).count();
                    whenPositive(hits,
                            v -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_HITS, v));
                    whenPositive(rj.size() - hits,
                            v -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_MISSES, v));
                }
            });

            return result;
        }
    }

    @Override
    public CompletionStage<Map<String, V>> getBulk
            (Function<Set<String>, Map<String, V>> factory, Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_GET_CALL, t));
             TimedFactory<String, V> timedFactory = new TimedFactory<>(factory, this::handleTimedFactory)) {
            final CompletionStage<Map<String, V>> result = getDelegate().getBulk(timedFactory, keys);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_GET, 1);
                } else {
                    final Map<String, V> rj = future.join();
                    whenPositive(rj.size() - timedFactory.getNumberOfKeys(),
                            v -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_HITS, v));
                    whenPositive(timedFactory.getNumberOfKeys(),
                            v -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_MISSES, v));
                }
            });

            return result;
        }
    }

    @Override
    public String getName() {
        return getDelegate().getName();
    }

    void handleTimedSupplier(Optional<Long> time) {
        if (time.isPresent()) {
            metricsRecorder.record(
                    getDelegate().getName(), EXTERNAL, TIMED_SUPPLIER_CALL, time.get());
        }
    }

    private void handleTimedFactory(Optional<Long> time, Long numberOfKeys) {
        if (time.isPresent()) {
            metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_FACTORY_CALL, time.get());
            metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FACTORY_KEYS, numberOfKeys);
        }
    }
}
