package com.atlassian.vcache.internal.core;

import com.atlassian.marshalling.api.MarshallingException;
import com.atlassian.marshalling.api.MarshallingPair;
import com.atlassian.vcache.ExternalCacheException;

import javax.annotation.Nullable;
import java.time.Duration;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.LongConsumer;

import static com.atlassian.vcache.ExternalCacheException.Reason.MARSHALLER_FAILURE;
import static java.util.Objects.requireNonNull;

/**
 * Contains common implementation methods.
 *
 * @since 1.0.0
 */
public class VCacheCoreUtils {
    /**
     * Returns a successfully completed {@link CompletionStage} with the specified result.
     *
     * @param value the result for the future
     * @param <T>   the type of the result
     * @return a successfully completed {@link CompletionStage} with the specified result.
     */
    public static <T> CompletionStage<T> successful(T value) {
        return CompletableFuture.completedFuture(value);
    }

    /**
     * Returns a failed {@link CompletionStage} with the specified cause.
     *
     * @param cause the cause of the failure
     * @param <T>   the type of the result
     * @param result A CompletionStage containing the result.
     *
     * @return a failed {@link CompletionStage} with the specified cause.
     */
    public static <T> CompletionStage<T> failed(CompletionStage<T> result, Throwable cause) {
        result.toCompletableFuture().completeExceptionally(cause);
        return result;
    }

    public static int roundUpToSeconds(Duration time) {
        if (time.isNegative()) {
            throw new IllegalArgumentException("duration cannot be negative: " + time);
        }

        final long result = (time.getNano() > 0) ? time.getSeconds() + 1 : time.getSeconds();

        if (result > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("duration exceeds maximum number that can be held in an int");
        }

        return (int) result;
    }

    public static void whenPositive(long number, LongConsumer handler) {
        if (number > 0) {
            handler.accept(number);
        }
    }

    public static <V> byte[] marshall(V data, MarshallingPair<V> valueMarshalling) throws ExternalCacheException {
        try {
            return valueMarshalling.getMarshaller().marshallToBytes(requireNonNull(data));
        } catch (MarshallingException e) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, e);
        }
    }

    public static <V> Optional<V> unmarshall(@Nullable byte[] data, MarshallingPair<V> valueMarshalling) throws ExternalCacheException {
        if (data == null) {
            return Optional.empty();
        }

        try {
            return Optional.of(valueMarshalling.getUnmarshaller().unmarshallFrom(data));
        } catch (MarshallingException ex) {
            throw new ExternalCacheException(MARSHALLER_FAILURE, ex);
        }
    }

    public static <K> boolean isEmpty(Iterable<K> iter) {
        return !iter.iterator().hasNext();
    }
}
