package com.atlassian.vcache.internal.core;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import static com.atlassian.vcache.internal.NameValidator.requireValidProductIdentifier;

/**
 * Represents the ability to generate the keys used with an external cache.
 *
 * @since 1.0.0
 */
public abstract class ExternalCacheKeyGenerator {
    private static final Logger log = LoggerFactory.getLogger(ExternalCacheKeyGenerator.class);

    private static final String SEPARATOR = "::";
    private static final String CACHE_VERSION_KEY = "cache-version";

    private final String productIdentifier;

    public ExternalCacheKeyGenerator(String productIdentifier) {
        this.productIdentifier = requireValidProductIdentifier(productIdentifier);
    }

    /**
     * Generates the key for an individual entry.
     */
    public String entryKey(String partition, String cacheName, long cacheVersion, String entryKey) {
        final String plain = productIdentifier + SEPARATOR + partition + SEPARATOR + cacheName
                + SEPARATOR + cacheVersion + SEPARATOR + entryKey;
        final String encoded = encode(plain);
        log.trace("Encoded cacheKey {} to {}", plain, encoded);
        return encoded;
    }

    /**
     * Generates the key used to hold the cache version number.
     */
    public String cacheVersionKey(String partition, String cacheName) {
        final String plain = productIdentifier + SEPARATOR + partition + SEPARATOR + cacheName
                + SEPARATOR + CACHE_VERSION_KEY;
        final String encoded = encode(plain);
        log.trace("Encoded cacheVersionKey {} to {}", plain, encoded);
        return encoded;
    }

    /**
     * Responsible for encoding the plain key using the appropriate algorithm.
     *
     * @param plain the plain key to be encoded
     * @return the key encoded
     */
    protected abstract String encode(String plain);
}
