package com.atlassian.vcache.internal.core.service;

import com.atlassian.vcache.internal.core.ExternalCacheKeyGenerator;
import io.atlassian.util.concurrent.Lazy;

import javax.annotation.Nullable;
import java.time.Duration;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Represents the request context for an {@link com.atlassian.vcache.ExternalCache} that maintains a
 * cache version.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
public class VersionedExternalCacheRequestContext<V> extends AbstractExternalCacheRequestContext<V> {
    protected final String externalCacheVersionKey;
    private final Supplier<Long> cacheVersionSupplier;

    @Nullable
    private Long cacheVersion;

    public VersionedExternalCacheRequestContext(ExternalCacheKeyGenerator keyGenerator,
                                                String name,
                                                Supplier<String> partitionSupplier,
                                                Function<String, Long> cacheVersionSupplier,
                                                Duration lockTimeout) {
        super(keyGenerator, name, partitionSupplier, lockTimeout);
        this.externalCacheVersionKey =
                keyGenerator.cacheVersionKey(partitionSupplier.get(), name);
        // memoize the cache version so we don't re-evaluate the function (and re-invoke the external cache) every time we need it
        this.cacheVersionSupplier = Lazy.supplier(() -> cacheVersionSupplier.apply(externalCacheVersionKey));
    }

    @Override
    protected long cacheVersion() {
        return cacheVersion == null
                ? cacheVersionSupplier.get()
                : cacheVersion;
    }

    public void updateCacheVersion(final Function<String, Long> cacheVersionSupplier) {
        this.cacheVersion = cacheVersionSupplier.apply(externalCacheVersionKey);
        clearKeyMaps();
    }
}
