package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.marshalling.api.MarshallingException;
import com.atlassian.marshalling.api.Unmarshaller;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_BYTES_UNMARSHALLED;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_UNMARSHALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_UNMARSHALL_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link Unmarshaller} that records metrics.
 *
 * @param <T> the value type
 * @since 1.0.0
 */
class TimedUnmarshaller<T> implements Unmarshaller<T> {
    private final Unmarshaller<T> delegate;
    private final MetricsRecorder metricsRecorder;
    private final String cacheName;

    TimedUnmarshaller(Unmarshaller<T> delegate, MetricsRecorder metricsRecorder, String cacheName) {
        this.delegate = requireNonNull(delegate);
        this.metricsRecorder = requireNonNull(metricsRecorder);
        this.cacheName = requireNonNull(cacheName);
    }

    @Override
    public T unmarshallFrom(byte[] raw) throws MarshallingException {
        metricsRecorder.record(cacheName, EXTERNAL, NUMBER_OF_BYTES_UNMARSHALLED, raw.length);
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(cacheName, EXTERNAL, TIMED_UNMARSHALL_CALL, t))) {
            return delegate.unmarshallFrom(raw);
        } catch (MarshallingException me) {
            metricsRecorder.record(cacheName, EXTERNAL, NUMBER_OF_FAILED_UNMARSHALL, 1);
            throw me;
        }
    }
}
