package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.PutPolicy;
import com.atlassian.vcache.TransactionalExternalCache;

import static com.atlassian.vcache.internal.MetricLabel.TIMED_PUT_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_ALL_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static java.util.Objects.requireNonNull;

/**
 * Wrapper for a {@link TransactionalExternalCache} that records metrics.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
class TimedTransactionalExternalCache<V>
        extends TimedExternalCache<V>
        implements TransactionalExternalCache<V> {
    private final TransactionalExternalCache<V> delegate;

    TimedTransactionalExternalCache(MetricsRecorder metricsRecorder, TransactionalExternalCache<V> delegate) {
        super(metricsRecorder);
        this.delegate = requireNonNull(delegate);
    }

    @Override
    protected TransactionalExternalCache<V> getDelegate() {
        return delegate;
    }

    @Override
    public void put(String key, V value, PutPolicy policy) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_PUT_CALL, t))) {
            getDelegate().put(key, value, policy);
        }
    }

    @Override
    public void remove(Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_REMOVE_CALL, t))) {
            getDelegate().remove(keys);
        }
    }

    @Override
    public void removeAll() {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_REMOVE_ALL_CALL, t))) {
            getDelegate().removeAll();
        }
    }
}
