package com.atlassian.vcache.internal.core.metrics;

import com.atlassian.vcache.ExternalWriteOperationsUnbuffered;
import com.atlassian.vcache.PutPolicy;

import java.util.concurrent.CompletionStage;

import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_PUT;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_REMOVE;
import static com.atlassian.vcache.internal.MetricLabel.NUMBER_OF_FAILED_REMOVE_ALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_PUT_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_ALL_CALL;
import static com.atlassian.vcache.internal.MetricLabel.TIMED_REMOVE_CALL;
import static com.atlassian.vcache.internal.core.metrics.CacheType.EXTERNAL;
import static com.atlassian.vcache.internal.core.metrics.TimedUtils.whenCompletableFuture;

/**
 * Wrapper for a {@link ExternalWriteOperationsUnbuffered} that records metrics.
 *
 * @param <V> the value type
 * @since 1.0.0
 */
abstract class TimedExternalWriteOperationsUnbuffered<V>
        extends TimedExternalCache<V>
        implements ExternalWriteOperationsUnbuffered<V> {
    TimedExternalWriteOperationsUnbuffered(MetricsRecorder metricsRecorder) {
        super(metricsRecorder);
    }

    protected abstract ExternalWriteOperationsUnbuffered<V> getDelegateOps();

    @Override
    public CompletionStage<Boolean> put(String key, V value, PutPolicy policy) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_PUT_CALL, t))) {
            final CompletionStage<Boolean> result = getDelegateOps().put(key, value, policy);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_PUT, 1);
                }
            });

            return result;
        }
    }

    @Override
    public CompletionStage<Void> remove(Iterable<String> keys) {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_REMOVE_CALL, t))) {
            final CompletionStage<Void> result = getDelegateOps().remove(keys);

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_REMOVE, 1);
                }
            });

            return result;
        }
    }

    @Override
    public CompletionStage<Void> removeAll() {
        try (ElapsedTimer ignored = new ElapsedTimer(
                t -> metricsRecorder.record(getDelegate().getName(), EXTERNAL, TIMED_REMOVE_ALL_CALL, t))) {
            final CompletionStage<Void> result = getDelegateOps().removeAll();

            whenCompletableFuture(result, future -> {
                if (future.isCompletedExceptionally()) {
                    metricsRecorder.record(getDelegate().getName(), EXTERNAL, NUMBER_OF_FAILED_REMOVE_ALL, 1);
                }
            });

            return result;
        }
    }
}
