package com.atlassian.vcache.marshallers;

import com.atlassian.annotations.Internal;
import com.atlassian.vcache.Marshaller;
import com.atlassian.vcache.MarshallerException;

import javax.annotation.Nonnull;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.CharacterCodingException;
import java.nio.charset.Charset;
import java.nio.charset.CharsetEncoder;
import java.nio.charset.CodingErrorAction;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;

/**
 * Optimised implementation for {@link String} objects.
 *
 * @since 1.0
 */
@Internal
class StringMarshaller implements Marshaller<String> {
    private static final Charset ENCODING_CHARSET = StandardCharsets.UTF_8;

    @Nonnull
    @Override
    public byte[] marshall(String str) throws MarshallerException {
        // Need to have the encoder complain if it cannot encode a character. By default it swallows the problem.
        final CharsetEncoder encoder = ENCODING_CHARSET.newEncoder();
        encoder.onMalformedInput(CodingErrorAction.REPORT);

        try {
            final ByteBuffer buffer = encoder.encode(CharBuffer.wrap(str));
            return Arrays.copyOf(buffer.array(), buffer.limit());
        } catch (CharacterCodingException e) {
            throw new MarshallerException("Unable to encode: " + str, e);
        }
    }

    @Nonnull
    @Override
    public String unmarshall(byte[] raw) {
        return new String(raw, ENCODING_CHARSET);
    }
}
