package com.atlassian.vcache;


import com.atlassian.annotations.PublicSpi;

import javax.annotation.Nonnull;

/**
 * Represents the ability to convert an object, of type <tt>T</tt>, to and from an array of {@code byte}s.
 * <p>Notes:</p>
 * <ul>
 * <li>
 * The {@link #unmarshall(byte[])} method should only ever be passed data that was generated by
 * calling the {@link #marshall(Object)} method on an instance of {@link Marshaller} associated
 * with a cache with the same <tt>name</tt>. However, the
 * {@link #unmarshall(byte[])} should endeavour to be robust in the event of arbitrary data.
 * </li>
 * <li>
 * Data versioning is the responsibility of the implementation.
 * </li>
 * <li>
 * The instances returned by {@link #unmarshall(byte[])} for the same input byte array must match when
 * compared using the {@link Object#equals(Object)} method.
 * </li>
 * <li>
 * See the <i>Data Considerations</i> section in the {@link ExternalCache} documentation.
 * </li>
 * <li>
 * The implementation must be multi-thread safe.
 * </li>
 * </ul>
 *
 * @param <T> the type that can be converted
 * @since 1.0
 */
@PublicSpi
public interface Marshaller<T> {
    /**
     * Responsible for converting a object into an array of bytes.
     *
     * @param obj the object to be converted.
     * @return the array of bytes representing <tt>data</tt>
     * @throws MarshallerException if unable to perform the conversion
     */
    @Nonnull
    byte[] marshall(T obj) throws MarshallerException;

    /**
     * Responsible for converting an array of bytes into an object. The instances returned for the same input byte
     * array must match when compared using the {@link Object#equals(Object)} method.
     *
     * @param raw the array of bytes to be converted.
     * @return the object represented by <tt>raw</tt>
     * @throws MarshallerException if unable to perform the conversion
     */
    @Nonnull
    T unmarshall(byte[] raw) throws MarshallerException;
}
