package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * Represents the common operations that can be performed on a JVM local cache.
 *
 * @param <K> the key type
 * @param <V> the value type
 * @since 1.0
 */
@PublicApi
public interface LocalCacheOperations<K, V> {
    /**
     * Returns a value that is associated with a specified key.
     *
     * @param key the key to check.
     * @return an {@link Optional} which may contain the value associated with the key.
     */
    @Nonnull
    Optional<V> get(K key);

    /**
     * Returns a value that may be associated with a specified key.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * If no entry for the specified key exists, then the specified <tt>supplier</tt> is called to create
     * an entry in the cache, before it is returned.
     * </li>
     * <li>
     * The <tt>supplier</tt> implementation needs to be multi-thread safe as it may be called concurrently
     * if multiple threads call this method.
     * </li>
     * </ul>
     *
     * @param key      the key uniquely identifying the value to be retrieved
     * @param supplier used to generate the value, if one does not exist already for the key. The supplier may not
     *                 return {@code null}.
     * @return the value associated with the key.
     */
    @Nonnull
    V get(K key, Supplier<? extends V> supplier);

    /**
     * Unconditionally puts the value under the specified key.
     *
     * @param key   the key to put the data under
     * @param value the value to associate with the key.
     */
    void put(K key, V value);

    /**
     * Conditionally puts the value under the specified key, if no entry currently exists. Returns the current
     * value associated with the key, if one currently exists.
     *
     * @param key   the key to put the data under
     * @param value the value to associate with the key.
     * @return {@link Optional#EMPTY} is no entry previously existed, otherwise the {@link Optional} contains
     * the current value.
     */
    @Nonnull
    Optional<V> putIfAbsent(K key, V value);

    /**
     * Conditionally replaces the value associated with a key, iff:
     * <ul>
     * <li>An entry already exists for the key</li>
     * <li>The current value matches the supplied <tt>currentValue</tt> using {@link Object#equals(Object)}</li>
     * </ul>
     *
     * @param key          the key to replace data under
     * @param currentValue the current value to match
     * @param newValue     the replacement value
     * @return whether the current value are replaced with the supplied <tt>newValue</tt>
     */
    boolean replaceIf(K key, V currentValue, V newValue);

    /**
     * Conditionally removed an entry for a specified key, iff:
     * <ul>
     * <li>An entry already exists for the key</li>
     * <li>The current value matches the supplied <tt>value</tt> using {@link Object#equals(Object)}</li>
     * </ul>
     *
     * @param key   the key of the entry to remove conditionally
     * @param value the current value to match
     * @return whether an entry was removed
     */
    boolean removeIf(K key, V value);

    /**
     * Removes the entries for the supplied key.
     *
     * @param key the key of the entry to remove
     */
    void remove(K key);

    /**
     * Removes all the entries in the cache.
     */
    void removeAll();
}
