package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import javax.annotation.Nonnull;
import java.util.Arrays;
import java.util.concurrent.CompletionStage;

/**
 * Represents the unbuffered write operations on an {@link ExternalCache}.
 *
 * @param <V> the value type
 * @since 1.0
 */
@PublicApi
public interface ExternalWriteOperationsUnbuffered<V> {
    /**
     * Puts the value under the specified key using the specified policy.
     *
     * @param key    the key to put the data under
     * @param value  the value to associate with the key.
     * @param policy the policy to apply
     * @return whether the entry was put into the cache.
     */
    @Nonnull
    CompletionStage<Boolean> put(String key, V value, PutPolicy policy);

    /**
     * Remove the entries with the specified keys.
     *
     * @param keys the keys of the entries to remove.
     * @return no indication is given as to whether the entries existed before they were deleted
     */
    @SuppressWarnings("unchecked")
    @Nonnull
    default CompletionStage<Void> remove(String... keys) {
        return remove(Arrays.asList(keys));
    }

    /**
     * Remove the entries with the specified keys.
     *
     * @param keys the keys of the entries to remove.
     * @return no indication is given as to whether the entries existed before they were deleted
     */
    @Nonnull
    CompletionStage<Void> remove(Iterable<String> keys);

    /**
     * Remove all entries in the cache.
     *
     * @return no indication is given as to whether entries existed before they were deleted
     */
    @Nonnull
    CompletionStage<Void> removeAll();
}
