package com.atlassian.vcache;

import com.atlassian.annotations.ExperimentalApi;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Provides a helper utilities.
 *
 * @since 1.0
 */
@ExperimentalApi
public class VCacheUtils {
    /**
     * Syntactic sugar for writing {@code stage.toCompletableFuture().unsafeJoin()}. This method should not be used
     * in production code, as it should be resilient to cache failures.
     *
     * @param stage the stage to call {@link CompletableFuture#join()} on
     * @param <V>   the value type
     * @return the result of joining.
     * @see CompletableFuture#join()
     * @see #unsafeJoin(CompletionStage)
     * @deprecated since 1.5.0 - use {@link #unsafeJoin(CompletionStage)} instead (if you must).
     */
    @Deprecated
    public static <V> V join(CompletionStage<V> stage) {
        return stage.toCompletableFuture().join();
    }

    /**
     * Syntactic sugar for writing {@code stage.toCompletableFuture().unsafeJoin()}. This method should not be used
     * in production code, as it should be resilient to cache failures. This method will throw runtime exceptions
     * if errors occur.
     *
     * @param stage the stage to call {@link CompletableFuture#join()} on
     * @param <V>   the value type
     * @return the result of joining.
     * @see CompletableFuture#join()
     */
    public static <V> V unsafeJoin(CompletionStage<V> stage) {
        return stage.toCompletableFuture().join();
    }

    /**
     * Syntactic sugar for handling the result (either success or failure) from a {@link CompletionStage}.
     *
     * @param stage   the stage to handle the results
     * @param success called if the stage returned a successful result
     * @param failure called if the stage failed
     * @param <T>     the type returned by the stage
     * @param <R>     the type to be returned
     * @return if the stage is successful, then the result of calling <tt>success</tt>, otherwise the result of
     * calling <tt>failure</tt>
     */
    public static <T, R> R fold(CompletionStage<T> stage, Function<T, R> success, Function<Throwable, R> failure) {
        return unsafeJoin(stage.handle((val, err) ->
                (err != null) ? failure.apply(err) : success.apply(val)));
    }

    /**
     * Syntactic sugar for writing {@code stage.handle(fn).toCompletableFuture().unsafeJoin()}.
     *
     * @param stage the stage to handle the results
     * @param fn    called to handle the result
     * @param <T>   the type returned by the stage
     * @param <R>   the type to be returned
     * @return the result of calling <tt>fn</tt> on the stage outcome.
     */
    public static <T, R> R fold(CompletionStage<T> stage, BiFunction<T, Throwable, R> fn) {
        return unsafeJoin(stage.handle(fn));
    }
}
