package com.atlassian.vcache;

import com.atlassian.annotations.PublicApi;

import java.time.Duration;
import java.util.Arrays;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletionStage;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Represents an external cache, where the data is ultimately stored in external cache systems
 * like Memcached or Redis. The values are stored by value and are converted using a
 * {@link com.atlassian.marshalling.api.MarshallingPair}.
 * <p>
 * The settings provided when creating a {@link ExternalCache} are only used within the JVM and are not
 * centrally stored across JVMs.
 * </p>
 * <h1>Failures</h1>
 * <p>
 * Failures are expected to happen when communicating with the external system, and the caller is expected to
 * deal with them. As such, all operations that involve communicating with the external system will return an
 * {@link CompletionStage}, which encapsulates that failure may have occurred.
 * </p>
 * <h1>WorkContext Lifecycle</h1>
 * <p>
 * Any returned {@link CompletionStage} is guaranteed to complete by the end of the Atlassian WorkContext it was
 * created within. There is no need to explicitly "join" on a {@link CompletionStage},
 * using {@link VCacheUtils#fold(CompletionStage, Function, Function)} or similar. This means that any "connected"
 * {@link CompletionStage}, via {@link CompletionStage#thenCombine(CompletionStage, BiFunction)} etc, will be
 * performed.
 * </p>
 * <h1>Buffering</h1>
 * <p>
 * Buffering of communications to the external cache system may be applied to both read and write operations.
 * The following three specialisations of the {@link ExternalCache} are provided to represent the possible
 * buffering modes:
 * </p>
 * <ul>
 * <li>{@link DirectExternalCache} - buffering is not applied for either read or write operations.</li>
 * <li>{@link TransactionalExternalCache} - buffering is applied for both read and write operations.</li>
 * <li>{@link StableReadExternalCache} - buffering is applied for only read operations</li>
 * </ul>
 * <p>
 * Buffering of read operations means that the result of every read operation
 * is stored in a private {@link RequestCache}. All read operations will first query the private
 * {@link RequestCache} to determine whether the operation has been performed before, and if it has, return the
 * <i>buffered</i> result.
 * </p>
 * <p>
 * Buffering of write operations means that all write operations are stored
 * in a private {@link RequestCache}, and at the end of the transaction they are applied to the
 * external cache system.
 * </p>
 * <p>
 * When buffering is used for write operations, the issue arises as to how to handle when there
 * are conflicts when replaying the buffered operations. The guiding principles are:
 * </p>
 * <ul>
 * <li>
 * The implementation shall endeavour to detect when the external system state has changed since it
 * was read and deal with it appropriately.
 * </li>
 * <li>
 * If the implementation cannot determine a reliable and consistent way to deal with a conflict,
 * it shall invalidate the entries affected, or the entire cache.
 * </li>
 * <li>
 * Rather than attempt to return errors (the implementation should log them) to the application code at
 * the end of a request, invalidation is used to recover.
 * </li>
 * </ul>
 * <p>
 * All read operations, even if buffered, may involve failures. As such, all read operations are encapsulated in
 * this interface. However, buffered write operations will not expose failures as they are replayed later. Hence,
 * the write operations are provided through the following two interfaces which are used by the
 * four specialisations of this interface:
 * </p>
 * <ul>
 * <li>{@link ExternalWriteOperationsBuffered} - encapsulates the buffered versions of the write operations.</li>
 * <li>{@link ExternalWriteOperationsUnbuffered} - encapsulates the unbuffered versions of the write operations.</li>
 * </ul>
 * <h1>Supplier/Factory Generation</h1>
 * <p>
 * The operations that may create values using a supplied {@link Supplier} or {@link Function}
 * make no guarantees about their atomicity. There is a possible race condition between detecting
 * that a value is missing in the external cache system, and the generated value being stored
 * in the external cache system. The implementations are free to override any (new) value that
 * may have been stored in the external cache system. If this behaviour is unacceptable, then
 * the caller is responsible for using a strategy to deal with this (such as global locking).
 * </p>
 * <h1>Time-to-live</h1>
 * <ul>
 * <li>
 * A cache has an associated time-to-live (<tt>ttl</tt>), which is expressed as {@link Duration}.
 * The implementations may round-up the supplied ttl to the nearest second. E.g. 500 milliseconds may be
 * converted to 1 second.
 * </li>
 * <li>
 * Implementations should implement ttl based on the time the entry was added or updated. However, do not assume that
 * the associated ttl defines exactly when entries are evicted. The implementations are free
 * to choose how the ttl is interpreted. It may expire entries in a timely manner, or it may be delayed until
 * next interaction with the cache.
 * </li>
 * </ul>
 * <h1>Data Considerations</h1>
 * <p>
 * As the data is stored externally to the JVM, the following needs to be considered:
 * </p>
 * <ul>
 * <li>
 * The data in the external system will persist longer than the JVM. Hence newer versions of the application
 * will need to handle data that could have been written by an earlier version of the application. If the data
 * format used is not compatible between the different versions of the application, problems will arise.
 * It is required that the application deal with potential versioning issues. One approach would be to
 * to use <a href="https://docs.oracle.com/javase/8/docs/platform/serialization/spec/version.html">
 * Versioning of Serializable Objects</a>.
 * </li>
 * <li>
 * Data is stored in the external system as byte arrays. The cache must be associated with
 * a {@link com.atlassian.marshalling.api.MarshallingPair} instance for marshalling values.
 * By extension, this means that an {@link ExternalCache} can only be used for data that can be marshalled
 * to/from byte arrays.
 * </li>
 * </ul>
 * <h1>Hints</h1>
 * <p>
 * Each cache is configured with the following hints to allow the implementation the opportunity to
 * optimise performance.
 * </p>
 * <ul>
 * <li>
 * {@link ExternalCacheSettings#getDataChangeRateHint()} - this is a hint on the expected rate of change to the
 * data held in the cache, including updating existing entries. A cache that holds values that are never
 * updated after being populated should use {@link ChangeRate#NONE}. A cache that holds data that changed
 * infrequently, like a cache of project details, should use {@link ChangeRate#LOW_CHANGE}.
 * </li>
 * <li>
 * {@link ExternalCacheSettings#getEntryGrowthRateHint()} - this is a hint on the expected rate of change in
 * the number of entries held in the cache. A cache that holds a fixed number of entries should
 * use {@link ChangeRate#NONE}.
 * </li>
 * <li>
 * {@link ExternalCacheSettings#getEntryCountHint()} - this is a hint on the desired number of entries in
 * the cache. The implementation is free to use this as it sees fit.
 * </li>
 * </ul>
 *
 * @param <V> the value type
 * @since 1.0
 */
@PublicApi
public interface ExternalCache<V> extends VCache {
    /**
     * Returns a value that is associated with a specified key.
     *
     * @param key the key to check.
     * @return an {@link Optional} which may contain the value associated with the key.
     */
    CompletionStage<Optional<V>> get(String key);

    /**
     * Returns a value that may be associated with a specified key.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * If no entry for the specified key exists, then the specified <tt>supplier</tt> is called. The value may be used
     * to create an entry in the cache. However, the value returned may be that of the supplier to a concurrent
     * call to this method, or that of the factory in a concurrent call to {@link #getBulk}.
     * When the supplier is called, callers may not assume that the value returned from the supplier is returned
     * by this method.
     * </li>
     * <li>
     * The <tt>supplier</tt> implementation needs to be multi-thread safe as it may be called concurrently
     * if multiple threads call this method.
     * </li>
     * </ul>
     *
     * @param key      the key uniquely identifying the value to be retrieved
     * @param supplier used to generate the value, if one does not exist already for the key. The supplier may not
     *                 return {@code null}.
     * @return the value associated with the key.
     */
    CompletionStage<V> get(String key, Supplier<V> supplier);

    /**
     * Returns the values that are associated with the specified keys. It is equivalent to calling
     * {@link #getBulk(Iterable)} passing {@code Arrays.asList(keys)} as the parameter.
     *
     * @param keys the keys to check.
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * {@link Optional} which may contain the value associated with the key.
     */
    @SuppressWarnings("unchecked")
    default CompletionStage<Map<String, Optional<V>>> getBulk(String... keys) {
        return getBulk(Arrays.asList(keys));
    }

    /**
     * Returns the values that are associated with the specified keys.
     *
     * @param keys the keys to check.
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * {@link Optional} which may contain the value associated with the key.
     */
    CompletionStage<Map<String, Optional<V>>> getBulk(Iterable<String> keys);

    /**
     * Returns the values that are associated with the specified keys. It is equivalent to calling
     * {@link #getBulk(Function, Iterable)} passing {@code factory} and {@code Arrays.asList(keys)} as the parameters.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * If no entry for a specified key (or keys) exists, then the specified <tt>factory</tt> is called. The values may
     * be used to create entries in the cache. However, the values returned may be that of the <tt>factory</tt>
     * to a concurrent call to this method, or that of the <tt>supplier</tt> in a concurrent call to {@link #get(String, Supplier)}.
     * When the <tt>factory</tt> is called, callers may not assume that the values returned from the <tt>factory</tt>
     * is returned by this method.
     * </li>
     * </ul>
     *
     * @param factory used to generate the values for the keys that do not have entries. The factory must return a
     *                map containing a non-null entry for each supplied key.
     * @param keys    the keys to retrieve
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * the value associated with the key.
     */
    @SuppressWarnings("unchecked")
    default CompletionStage<Map<String, V>> getBulk(Function<Set<String>, Map<String, V>> factory, String... keys) {
        return getBulk(factory, Arrays.asList(keys));
    }

    /**
     * Returns the values that are associated with the specified keys.
     * <p>Notes:</p>
     * <ul>
     * <li>
     * If no entry for a specified key (or keys) exists, then the specified <tt>factory</tt> is called. The values may
     * be used to create entries in the cache. However, the values returned may be that of the <tt>factory</tt>
     * to a concurrent call to this method, or that of the <tt>supplier</tt> in a concurrent call to {@link #get(String, Supplier)}.
     * When the <tt>factory</tt> is called, callers may not assume that the values returned from the <tt>factory</tt>
     * is returned by this method.
     * </li>
     * </ul>
     *
     * @param factory used to generate the values for the keys that do not have entries. The factory must return a
     *                map containing a non-null entry for each supplied key.
     * @param keys    the keys to retrieve
     * @return A {@link Map} that is keyed on the {@code keys} specified. Each entry in the {@link Map} will have
     * the value associated with the key.
     */
    CompletionStage<Map<String, V>> getBulk(Function<Set<String>, Map<String, V>> factory, Iterable<String> keys);
}
