package com.atlassian.utils.process;

import org.apache.log4j.Logger;
import org.jvnet.winp.WinProcess;

import java.io.File;
import java.io.IOException;
import java.util.*;
import java.util.concurrent.*;

/**
 * This class manages the execution of an external process, using separate threads to process
 * the process' IO requirements.
 */
public class ExternalProcessImpl implements ExternalProcess {

    private static final Logger LOG = Logger.getLogger(ExternalProcessImpl.class);
    private static final String OS_NAME = System.getProperty("os.name").toLowerCase();
    private static final ExecutorService POOL;

    private boolean canceled;
    private List<String> command;
    private Map<String, String> environment;
    private LatchedRunnable errorPump;
    private long executionTimeout;
    private ProcessHandler handler;
    private long idleTimeout;
    private LatchedRunnable inputPump;
    private long lastWatchdogReset;
    private List<ProcessMonitor> monitors;
    private LatchedRunnable outputPump;
    private Process process;
    private ProcessException processException;
    private boolean suppressSpecialWindowsBehaviour;
    private long startTime;
    private File workingDir;
    private boolean useWindowsEncodingWorkaround;

    static {
        final String pooledThreadName = "ExtProcess IO Pump";
        ThreadFactory threadFactory = new ThreadFactory() {
            public Thread newThread(Runnable r) {
                return new Thread(r, pooledThreadName);
            }
        };
        POOL = new ThreadPoolExecutor(6, Integer.MAX_VALUE, 2, TimeUnit.MINUTES,
                new SynchronousQueue<Runnable>(), threadFactory) {

            @Override
            protected void beforeExecute(Thread thread, Runnable runnable) {
                thread.setName(thread.getId() + ":" + ((LatchedRunnable) runnable).getName());
                super.beforeExecute(thread, runnable);
            }

            @Override
            protected void afterExecute(Runnable runnable, Throwable throwable) {
                Thread.currentThread().setName(pooledThreadName);
                super.afterExecute(runnable, throwable);
            }
        };
    }

    /**
     * Process an external command.
     *
     * @param command the command and its arguments as separate elements
     * @param handler the process handler to manage the execution of this process
     */
    public ExternalProcessImpl(String[] command, ProcessHandler handler) {
        this(Arrays.asList(command), handler);
    }

    /**
     * Process an external command.
     *
     * @param command the command and its arguments as separate elements
     * @param handler the process handler to manage the execution of this process
     */
    public ExternalProcessImpl(List<String> command, ProcessHandler handler) {
        setCommand(command);
        setHandler(handler);

        idleTimeout = TimeUnit.MINUTES.toMillis(1L);
        monitors = new ArrayList<ProcessMonitor>();
        startTime = -1;
    }

    /**
     * Process an external command. The command is given as a single command line and parsed into
     * the command and its arguments. Spaces are used as argument delimiters so if any command arguments
     * need to contain spaces, the array or list based constructors should be used.
     *
     * @param commandLine the command and its arguments in a single line. If any arguments
     *                    need to contain spaces, the array or list based constructors should be used.
     * @param handler     The handler for this execution. The handler supports the required IO
     *                    operations
     */
    public ExternalProcessImpl(String commandLine, ProcessHandler handler) {
        this(ProcessUtils.tokenizeCommand(commandLine), handler);
    }

    public void resetWatchdog() {
        lastWatchdogReset = System.currentTimeMillis();
    }

    public long getTimeoutTime() {
        long timeout = lastWatchdogReset + idleTimeout;
        if (executionTimeout > 0 && startTime > 0) {
            timeout = Math.min(timeout, startTime + executionTimeout);
        }
        return timeout;
    }

    public boolean isTimedOut() {
        return getTimeoutTime() < System.currentTimeMillis();
    }

    protected void setHandler(ProcessHandler handler) {
        this.handler = handler;
    }

    private void setCommand(List<String> command) {
        this.command = command;
    }

    public void setWorkingDir(File workingDir) {
        this.workingDir = workingDir;
    }

    public void setEnvironment(Map<String, String> environment) {
        this.environment = environment;
    }

    public void setSuppressSpecialWindowsBehaviour(boolean suppressSpecialWindowsBehaviour) {
        this.suppressSpecialWindowsBehaviour = suppressSpecialWindowsBehaviour;
    }

    public void setUseWindowsEncodingWorkaround(boolean useWindowsEncodingWorkaround) {
        this.useWindowsEncodingWorkaround = useWindowsEncodingWorkaround;
    }

    private boolean arePumpsRunning() {
        return outputPump.isRunning() || errorPump.isRunning()
                || (inputPump != null && inputPump.isRunning());
    }

    /**
     * Get the process handler for this process execution
     *
     * @return the ProcessHandler instance associated with this process execution.
     */
    public ProcessHandler getHandler() {
        return handler;
    }

    /**
     * @return the time process execution started. -1 if the process has not yet started.
     */
    public long getStartTime() {
        return this.startTime;
    }

    public void addMonitor(ProcessMonitor monitor) {
        this.monitors.add(monitor);
    }

    public void removeMonitor(ProcessMonitor monitor) {
        this.monitors.remove(monitor);
    }

    private boolean isWindows() {
        return OS_NAME.contains("windows");
    }

    private String quoteString(String value) {
        StringBuilder builder = new StringBuilder()
                .append("\"")
                .append(value.replace("\"", "\\\""))
                .append("\"");
        return builder.toString();
    }

    /*
      * This method provides a workaround for a JVM bug on windows (see http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4947220). The bug
      * is that the Sun/Oracle JVM uses the (8 bit) OEM codepage for encoding commandline arguments that are passed to an external process.  Any
      * characters in a command line argument that can't be represented in the OEM codepage will be replaced by the '?' character, which will probably
      * cause the command that's being called to fail.
      *
      * A bit of background information is helpful to understand what's going on. Windows uses 2 code pages: the OEM code page and the ANSI code page
      * (or 'Windows code page'). The OEM code page is always limited to 8 bit character encodings, whereas some windows code pages are 8 bit and some
      * are larger. The OEM code page is typically used for console applications, whereas the windows code pages are used for native non-Unicode application
      * using a GUI on Windows systems.  The system-wide settings can be found in the windows registry:
      *
      * More info about the history of OEM vs ANSI code pages: http://blogs.msdn.com/b/michkap/archive/2005/02/08/369197.aspx
      *
      * The workaround is to store the command-line arguments in environment variables and refer to the environment vars on the command line. Windows
      * cmd will expand the command line arguments without using to the OEM code page, which usually has more correlation with the locale of the
      * producer of the hardware (it's derived from the BIOS code page) than with the regional settings configured in Windows. The ANSI code page is derived from
      * the regional settings.
      *
      * When cmd expands the %JENV_XXX% vars on the command line it uses the ANSI code page instead of the OEM code page (or that is what testing
      * seems to indicate, can't find any definitive answer in the cmd docos). While this still isn't a true fix to the problem, for most situations this will be sufficient
      * as the user typically won't use characters that aren't defined for his locale. But then again, they might..
     */
    private Process createWinProcess(List<String> command, Map<String, String> environment, File workingDir)
            throws IOException {
        List<String> newCommand = new ArrayList<String>(command.size() + 3);
        newCommand.add("cmd");
        newCommand.add("/A"); // use ANSI encoding
        newCommand.add("/C");

        if (useWindowsEncodingWorkaround) {
            newCommand.add(command.get(0));

            Map<String, String> i18nEnvironment = environment == null ? new HashMap<String, String>() : new HashMap<String, String>(environment);
            for (int counter = 1; counter < command.size(); counter++) {
                String envName = "JENV_" + counter;
                newCommand.add("%" + envName + "%");
                i18nEnvironment.put(envName, quoteString(command.get(counter)));
            }
            environment = i18nEnvironment;
        } else {
            newCommand.addAll(command);
        }

        ProcessBuilder builder = new ProcessBuilder(newCommand)
                .directory(workingDir);
        if (environment != null) {
            builder.environment().putAll(environment);
        }
        if (LOG.isDebugEnabled()) {
            logProcessDetails(builder);
        }
        return builder.start();
    }

    protected Process createProcess(List<String> command, Map<String, String> environment, File workingDir)
            throws IOException {
        if (!suppressSpecialWindowsBehaviour && isWindows()) {
            return createWinProcess(command, environment, workingDir);
        } else {
            ProcessBuilder builder = new ProcessBuilder(command).directory(workingDir);
            if (environment != null) {
                builder.environment().putAll(environment);
            }
            if (LOG.isDebugEnabled()) {
                logProcessDetails(builder);
            }
            return builder.start();
        }
    }

    private void logProcessDetails(ProcessBuilder processBuilder) {
        String divider = "---------------------------";
        LOG.debug(divider);
        LOG.debug("Start Process Debug Information");
        LOG.debug(divider);
        LOG.debug("Command");
        LOG.debug(processBuilder.command());
        LOG.debug(divider);
        LOG.debug("Working Dir");
        LOG.debug(processBuilder.directory());
        LOG.debug(divider);
        LOG.debug("Environment");
        for (Map.Entry entry : processBuilder.environment().entrySet()) {
            LOG.debug(entry.getKey() + ": " + entry.getValue());
        }
        LOG.debug(divider);
        LOG.debug("Redirect Error Stream?");
        LOG.debug(processBuilder.redirectErrorStream());
        LOG.debug(divider);
        LOG.debug("End Process Debug Information");
        LOG.debug(divider);
    }

    /**
     * Start the external process and setup the IO pump threads needed to
     * manage the process IO. If you call this method you must eventually call the
     * finish() method. Using this method you may execute additional code between process
     * start and finish.
     */
    public void start() {
        try {
            this.startTime = System.currentTimeMillis();
            this.process = createProcess(command, environment, workingDir);
            setupIOPumps();
        } catch (IOException e) {
            processException = new ProcessException(e);
        }
    }

    private void handleHandlerError(String handlerName, Throwable t) {
        if (!isCanceled()) {
            LOG.debug(handlerName + " encountered an error; aborting process", t);
            cancel();
            if (t instanceof ProcessException) {
                processException = (ProcessException) t;
            } else {
                processException = new ProcessException(t);
            }
        } else {
            LOG.debug(handlerName + ": Process canceled; ignoring exception", t);
        }
    }
    
    private void setupIOPumps() {
        // set up threads to feed data to and extract data from the process
        if (handler.hasInput()) {
            inputPump = new LatchedRunnable("StdInHandler " + process) {

                protected void doTask() {
                    try {
                        handler.provideInput(process.getOutputStream());
                    } catch (Throwable t) {
                        handleHandlerError(name, t);
                    }
                }
            };
        }

        errorPump = new LatchedRunnable("StdErrHandler " + process) {

            protected void doTask() {
                try {
                    handler.processError(process.getErrorStream());
                } catch (Throwable t) {
                    handleHandlerError(name, t);
                }
            }
        };

        outputPump = new LatchedRunnable("StdOutHandler " + process) {

            protected void doTask() {
                try {
                    handler.processOutput(process.getInputStream());
                } catch (Throwable t) {
                    handleHandlerError(name, t);
                }
            }
        };

        // tickle the dog initially
        resetWatchdog();
        handler.setWatchdog(this);

        POOL.execute(errorPump);
        POOL.execute(outputPump);
        if (inputPump != null) {
            POOL.execute(inputPump);
        }
    }

    /**
     * Finish process execution. This method should be called after you have called the
     * start() method.
     */
    public void finish() {
        if (process != null) {
            try {
                do {
                    long checkTime = getTimeoutTime();
                    awaitPump(outputPump, checkTime);
                    awaitPump(inputPump, checkTime);
                    awaitPump(errorPump, checkTime);
                } while (!isTimedOut() && arePumpsRunning() && !Thread.currentThread().isInterrupted());
            } finally {
                if (Thread.currentThread().isInterrupted()) {
                    cancel();

                    // All is good, now clearing interrupted state of current thread.
                    Thread.interrupted();
                }

                int exitCode = wrapUpProcess();
                handler.complete(exitCode, canceled, processException);
            }
        } else {
            handler.complete(-1, false, processException);
        }
    }

    /**
     * Notifies all ProcessMonitors of the 'beforeStart' event.
     */
    private void notifyBeforeStart() {
        for (ProcessMonitor monitor : monitors) {
            try {
                monitor.onBeforeStart(this);
            } catch (Exception e) {
                // catch and log error, but continue
                LOG.error("Error while processing 'beforeStarted' event:", e);
            }
        }
    }

    /**
     * Notifies all ProcessMonitors of the 'afterFinished' event.
     */
    private void notifyAfterFinished() {
        for (ProcessMonitor monitor : monitors) {
            try {
                monitor.onAfterFinished(this);
            } catch (Exception e) {
                LOG.error("Error while processing 'afterFinished' event:", e);
            }
        }
    }

    /**
     * Execute the external command. When this method returns, the process handler
     * provided at construction time should be consulted to collect exit code, exceptions,
     * process output, etc.
     */
    public void execute() {
        notifyBeforeStart();
        try {
            start();
            finish();
        } finally {
            notifyAfterFinished();
        }
    }

    /**
     * Executes the external command. While it is running, the given runnable is executed.
     * The external command is not checked until the runnable completes
     *
     * @param runnable A task to perform while the external command is running.
     */
    public void executeWhile(Runnable runnable) {
        start();
        if (runnable != null) {
            runnable.run();
        }
        finish();
    }

    public String getCommandLine() {
        StringBuilder builder = new StringBuilder();
        for (String s : command) {
            if (builder.length() > 0) {
                builder.append(" ");
            }
            builder.append(s);
        }
        return builder.toString();
    }

    /**
     * Wait a given time for the process to finish
     *
     * @param maxWait the maximum amount of time in milliseconds to wait for the process to finish
     * @return true if the process has finished.
     */
    public boolean finish(int maxWait) {
        if (process != null) {
            boolean finished = false;
            try {
                long endTime = System.currentTimeMillis() + maxWait;
                awaitPump(outputPump, endTime);
                awaitPump(inputPump, endTime);
                awaitPump(errorPump, endTime);
            } finally {
                if (!arePumpsRunning()) {
                    // process finished
                    finished = true;
                    int exitCode = wrapUpProcess();
                    handler.complete(exitCode, canceled, processException);
                }
            }
            return finished;
        } else {
            handler.complete(-1, false, processException);
            return true;
        }
    }

    private int wrapUpProcess() {
        int exitCode = -1;
        boolean processIncomplete = true;
        boolean interrupted = false;
        try {
            exitCode = process.exitValue();
            processIncomplete = false;
        } catch (IllegalThreadStateException itse) {
            // process still running - could be a race to have the process finish so wait a little to be sure
            while (processIncomplete && System.currentTimeMillis() - getTimeoutTime() < 10) {
                // we are currently before the end of the period (within 10ms slack), so process probably not ready yet
                try {
                    Thread.sleep(100);
                    exitCode = process.exitValue();
                    processIncomplete = false;
                } catch (InterruptedException ie) {
                    processIncomplete = true;
                    interrupted = true;
                    break;
                } catch (IllegalThreadStateException e) {
                    // ignore and try in the next loop
                }
            }
        } finally {
            internalCancel();
        }

        if (processIncomplete && !interrupted) {
            processException = new ProcessTimeoutException("process timed out");
        }
        return exitCode;
    }

    private void awaitPump(LatchedRunnable runnable, long latestTime) {
        if (runnable != null) {
            long timeout = latestTime - System.currentTimeMillis();
            if (timeout < 1) {
                timeout = 1;
            }
            runnable.await(timeout);
        }
    }

    /**
     * Cancel should be called if you wish to interrupt process execution.
     */
    public void cancel() {
        this.canceled = true;

        internalCancel();
    }

    private void internalCancel() {
        if (outputPump != null) {
            outputPump.cancel();
        }
        if (inputPump != null) {
            inputPump.cancel();
        }
        if (errorPump != null) {
            errorPump.cancel();
        }
        if (process != null) {
            if (isWindows()) {
                try {
                    new WinProcess(process).killRecursively();
                } catch (Throwable t) {
                    LOG.error("Failed to kill Windows process; falling back on Process.destroy()", t);
                    process.destroy();
                }
            } else {
                process.destroy();
            }
        }
    }

    public boolean isCanceled() {
        return canceled;
    }

    public void setExecutionTimeout(long executionTimeout) {
        this.executionTimeout = executionTimeout;
    }

    public void setIdleTimeout(long idleTimeout) {
        this.idleTimeout = idleTimeout;
    }

    @Deprecated
    public void setTimeout(long timeout) {
        setIdleTimeout(timeout);
    }

    /**
     * Attempts to shutdown the internal {@code ThreadPoolExecutor} which manages the I/O pumps for external processes.
     * To prevent memory leaks, web applications which use process-utils should always call this when they terminate.
     * <p/>
     * On termination, an attempt is made to shutdown the thread pool gracefully. If that does not complete within five
     * seconds, shutdown is forced. That is given another five seconds to complete before the thread pool is abandoned.
     *
     * @since 1.5
     */
    public static void shutdown() {
        if (POOL == null) {
            return;
        }
        LOG.debug("Attempting to shutdown pump executor service");

        POOL.shutdown();
        try {
            if (POOL.awaitTermination(5, TimeUnit.SECONDS)) {
                LOG.debug("Pump executor service has shutdown gracefully");
            } else {
                //The executor did not shutdown within the timeout. We can't wait forever, though, so issue a
                //shutdownNow() and give it another second
                LOG.warn("Pump executor service did not shutdown within the timeout; forcing shutdown");

                POOL.shutdownNow();
                if (POOL.awaitTermination(5, TimeUnit.SECONDS)) {
                    //The forced shutdown has brought the executor down. Not ideal, but acceptable
                    LOG.debug("Pump executor service has been forced to shutdown");
                } else {
                    //We can't delay execution indefinitely waiting, so log a warning. The JVM may not shut down
                    //if this service does not stop (because it uses non-daemon threads), so this may be helpful
                    //in debugging should that happen.
                    LOG.warn("Pump executor service did not shutdown; it will be abandoned");
                }
            }
        } catch (InterruptedException e) {
            LOG.warn("Interrupted while waiting for the pump executor service to shutdown; some worker threads may " +
                    "still be running");
        }
    }
}
