package com.atlassian.utils.process;

import org.apache.log4j.Logger;

import java.io.InputStream;
import java.io.OutputStream;

/**
 * An implementation of a ProcessHandler where the processors for the process' IO streams can
 * be plugged in.
 *
 */
public class PluggableProcessHandler extends BaseProcessHandler {
    private int exitCode;
    private ProcessException exception;

    private OutputHandler outputHandler;
    private OutputHandler errorHandler;
    private InputHandler inputHandler;

    private boolean complete;
    private boolean throwOnNonZeroExit = true;

    private static final Logger LOGGER = Logger.getLogger(PluggableProcessHandler.class);

    public PluggableProcessHandler() {
    }

    public void processOutput(InputStream processOutput) throws ProcessException {
        if (outputHandler == null) {
            throw new IllegalStateException("Process output received with no handler");
        }
        outputHandler.setWatchdog(getWatchdog());
        outputHandler.process(processOutput);
    }

    public void processError(InputStream processError) throws ProcessException {
        if (errorHandler == null) {
            throw new IllegalStateException("Process error output received with no handler");
        }
        errorHandler.setWatchdog(getWatchdog());
        errorHandler.process(processError);
    }

    public boolean hasInput() {
        return inputHandler != null;
    }

    public void provideInput(OutputStream processInput) {
        if (!hasInput()) {
            throw new IllegalStateException("Attempt to read input without an input handler");
        }
        inputHandler.setWatchdog(getWatchdog());
        inputHandler.process(processInput);
    }

    public void complete(int code, ProcessException pe) {
        this.exitCode = code;
        this.exception = pe;
        if (outputHandler != null) {
            try {
                outputHandler.complete();
            } catch (Throwable e) {
                setException(e);
            }
        }
        if (errorHandler != null) {
            try {
                errorHandler.complete();
            } catch (Throwable e) {
                setException(e);
            }
        }
        if (inputHandler != null) {
            try {
                inputHandler.complete();
            } catch (Throwable e) {
                setException(e);
            }
        }

        if (pe == null && code != 0 && throwOnNonZeroExit) {
            setException(new ProcessException("Non-zero exit code: " + code, code));
        }
        complete = true;
    }

    public int getExitCode() {
        return exitCode;
    }

    public ProcessException getException() {
        return exception;
    }

    public void setException(Throwable e) {
        if (exception == null) {
            if (e instanceof ProcessException) {
                this.exception = (ProcessException) e;
            } else {
                this.exception = new ProcessException(e);
            }
        } else {
            LOGGER.debug("Ignored exception as exception for handler is already set", e);
        }
    }

    public String getError() {
        return null;
    }

    public void setOutputHandler(OutputHandler outputHandler) {
        this.outputHandler = outputHandler;
    }

    public void setErrorHandler(OutputHandler errorHandler) {
        this.errorHandler = errorHandler;
    }

    public void setInputHandler(InputHandler inputHandler) {
        this.inputHandler = inputHandler;
    }

    public OutputHandler getOutputHandler() {
        return outputHandler;
    }

    public OutputHandler getErrorHandler() {
        return errorHandler;
    }

    public InputHandler getInputHandler() {
        return inputHandler;
    }

    public boolean succeeded() {
        return exception == null;
    }

    public void reset() {
        exitCode = 0;
        exception = null;
        complete = false;
    }

    public boolean isComplete() {
        return complete;
    }

    public boolean isThrowOnNonZeroExit() {
        return throwOnNonZeroExit;
    }

    public void setThrowOnNonZeroExit(boolean throwOnNonZeroExit) {
        this.throwOnNonZeroExit = throwOnNonZeroExit;
    }
}
