package com.atlassian.utils.process;

import org.apache.log4j.Logger;
import org.apache.log4j.NDC;

import java.util.Stack;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

/**
 * Date: 16/07/2008
 * Time: 14:54:03
 */
public abstract class LatchedRunnable implements Runnable {
    private static Logger log = Logger.getLogger(LatchedRunnable.class);
    private CountDownLatch latch = new CountDownLatch(1);
    private Thread runner;
    private boolean cancelled;
    private Stack<?> ndcStack = NDC.cloneStack();
    protected String name;

    protected LatchedRunnable()
    {
    }

    protected LatchedRunnable(final String name)
    {
        this.name = name;
    }

    public final void run() {
        try {
            NDC.inherit(ndcStack);
            runner = Thread.currentThread();
            doTask();
        } finally {
            latch.countDown();
            NDC.remove();
        }
    }

    public boolean await(long millis) {
        try {
            return latch.await(millis, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            log.warn("Interrupted waiting for ExternalProcess pump to complete");
            Thread.currentThread().interrupt();
            return false;
        }
    }

    public boolean isRunning() {
        return latch.getCount() > 0;
    }

    protected abstract void doTask();


    public void interrupt() {
        if (runner != null) {
            runner.interrupt();
        }
    }

    public void cancel() {
        this.cancelled = true;

        // Generally pointless, giving likely underlying blocking io, but logically correct in any case. Technically, doTask could do anything
        interrupt();
    }

    public boolean isCancelled() {
        return cancelled;
    }
}