package com.atlassian.utils.process;

import org.apache.log4j.Logger;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

/**
 * This class manages the execution of an external process, using separate threads to process
 * the process' IO requirements.
 */
public class ExternalProcess implements Watchdog {
    private static final Logger log = Logger.getLogger(ExternalProcess.class);
    private static final String OS_NAME = System.getProperty("os.name");
    
    private boolean useWindowsEncodingWorkaround = false;
    private List<String> command;
    private File workingDir;
    private Map<String, String> environment;
    private ProcessHandler handler;
    private Process process;

    private final List<ProcessMonitor> monitors = new ArrayList<ProcessMonitor>();
    
    private ProcessException processException;

    private LatchedRunnable outputPump;
    private LatchedRunnable errorPump;
    private LatchedRunnable inputPump;

    private static final ExecutorService pumpThreadPool;

    private long lastWatchdogReset;
    private long timeout = 60000L;
    private Long startTime;
    private boolean cancelled;
    
    public void resetWatchdog() {
        lastWatchdogReset = System.currentTimeMillis();
    }

    public long getTimeoutTime() {
        return lastWatchdogReset + timeout;
    }

    public boolean isTimedOut() {
        return getTimeoutTime() < System.currentTimeMillis();
    }

    static {
        ThreadFactory threadFactory = new ThreadFactory() {
            public Thread newThread(Runnable r) {
                return new Thread(r, "ExtProcess IO Pump");
            }
        };
        pumpThreadPool = new ThreadPoolExecutor(6, Integer.MAX_VALUE, 120, TimeUnit.SECONDS,
                                                new SynchronousQueue<Runnable>(), threadFactory);
    }

    /**
     * Process an external command.
     * @param command the command and its arguments as separate elements
     * @param handler The handler for this execution. The handler supports the required IO
     *                operations
     */
    public ExternalProcess(String[] command, ProcessHandler handler) {
        setCommand(Arrays.asList(command));
        setHandler(handler);
    }

    /**
     * Process an external command (the command and arguments are given as a list)
     * @param command A list containing the command and its arguments
     * @param handler The process handler to manage the execution of this process.
     */
    public ExternalProcess(List<String> command, ProcessHandler handler) {
        setCommand(command);
        setHandler(handler);
    }

    /**
     * Process an external command. The command is given as a single command line and parsed into
     * the command and its arguments. Spaces are used as argument delimiters so if any command arguments
     * need to contain spaces, the array or list based constructors should be used.
     *
     * @param commandLine the command and its arguments in a single line. If any arguments
     *                    need to contain spaces, the array or list based constructors should be used.
     * @param handler The handler for this execution. The handler supports the required IO
     *                operations
     */
    public ExternalProcess(String commandLine, ProcessHandler handler) {
        String[] cmdArray = ProcessUtils.tokenizeCommand(commandLine);
        setCommand(Arrays.asList(cmdArray));
        setHandler(handler);
    }

    private void setHandler(ProcessHandler handler) {
        this.handler = handler;
    }

    private void setCommand(List<String> command) {
        this.command = command;
    }

    public void setWorkingDir(File workingDir) {
        this.workingDir = workingDir;
    }

    public void setEnvironment(Map<String, String> environment) {
        this.environment = environment;
    }

    private boolean arePumpsRunning() {
        return outputPump.isRunning() || errorPump.isRunning()
                || (inputPump != null && inputPump.isRunning());
    }

    public void setUseWindowsEncodingWorkaround(final boolean useWindowsEncodingWorkaround) {
        this.useWindowsEncodingWorkaround = useWindowsEncodingWorkaround;
    }

    /**
     * Get the process handler for this process execution
     *
     * @return the ProcessHandler instance associated with this process execution.
     */
    public ProcessHandler getHandler() {
        return handler;
    }

    /**
     * @return the time process execution started. null if the process has not yet started.
     */
    public Long getStartTime() {
        return this.startTime;
    }
    
    public void addMonitor(ProcessMonitor monitor) {
        this.monitors.add(monitor);
    }
    
    public void removeMonitor(ProcessMonitor monitor) {
        this.monitors.remove(monitor);
    }
    
    private boolean isWindows() {
    	return OS_NAME.toLowerCase().contains("windows");
    }
    
    private String quoteString(String value) {
        StringBuilder builder = new StringBuilder()
            .append("\"")
            .append(value.replace("\"", "\\\""))
            .append("\"");
        return builder.toString();
    }
    
    /*
      * This method provides a workaround for a JVM bug on windows (see http://bugs.sun.com/bugdatabase/view_bug.do?bug_id=4947220). The bug
      * is that the Sun/Oracle JVM uses the (8 bit) OEM codepage for encoding commandline arguments that are passed to an external process.  Any
      * characters in a command line argument that can't be represented in the OEM codepage will be replaced by the '?' character, which will probably
      * cause the command that's being called to fail.
      * 
      * A bit of background information is helpful to understand what's going on. Windows uses 2 code pages: the OEM code page and the ANSI code page 
      * (or 'Windows code page'). The OEM code page is always limited to 8 bit character encodings, whereas some windows code pages are 8 bit and some
      * are larger. The OEM code page is typically used for console applications, whereas the windows code pages are used for native non-Unicode application
      * using a GUI on Windows systems.  The system-wide settings can be found in the windows registry:
      * 
      * 
      * More info about the history of OEM vs ANSI code pages: http://blogs.msdn.com/b/michkap/archive/2005/02/08/369197.aspx
      * 
      * The workaround is to store the command-line arguments in environment variables and refer to the environment vars on the command line. Windows
      * cmd will expand the command line arguments without using to the OEM code page, which usually has more correlation with the locale of the 
      * producer of the hardware (it's derived from the BIOS code page) than with the regional settings configured in Windows. The ANSI code page is derived from
      * the regional settings.
      * 
      * When cmd expands the %JENV_XXX% vars on the command line it uses the ANSI code page instead of the OEM code page (or that is what testing
      * seems to indicate, can't find any definitive answer in the cmd docos). While this still isn't a true fix to the problem, for most situations this will be sufficient 
      * as the user typically won't use characters that aren't defined for his locale. But then again, they might..
     */
    private Process createWinProcess(List<String> command, Map<String, String> environment, File workingDir) throws IOException {
        final List<String> newCommand = new ArrayList<String>();
        newCommand.add("cmd");
        newCommand.add("/A");
        newCommand.add("/C");
        newCommand.add("call");

        if (useWindowsEncodingWorkaround) {
            Map<String, String> newEnv = environment != null ? new HashMap<String, String>(environment) : new HashMap<String, String>();
            for (int counter = 1; counter < command.size(); counter++) {
                final String envName = "JENV_" + counter;
                newCommand.add("%" + envName + "%");
                newEnv.put(envName, quoteString(command.get(counter)));
            }
            environment = newEnv;
        } else {
            newCommand.addAll(command);
        }

        ProcessBuilder pb = new ProcessBuilder(newCommand);
        pb.directory(workingDir);
        if (environment != null)
        {
            pb.environment().putAll(environment);
        }

        if (log.isDebugEnabled())
        {
            logProcessDetails(pb);
        }

        return pb.start();
    }

    private Process createProcess(List<String> command, Map<String, String> environment, File workingDir) throws IOException {
    	if (isWindows()) {
    		return createWinProcess(command, environment, workingDir);
    	} else {
            //But what about BeOS?!
            final ProcessBuilder processBuilder = new ProcessBuilder()
                    .command(command)
                    .directory(workingDir);

            if (environment != null)
            {
                processBuilder.environment().putAll(environment);
            }

            if (log.isDebugEnabled())
            {
                logProcessDetails(processBuilder);
            }

            return processBuilder.start();
    	}
    }

    private void logProcessDetails(ProcessBuilder processBuilder)
    {
        String divider = "---------------------------";
        log.debug(divider);
        log.debug("Start Process Debug Information");
        log.debug(divider);
        log.debug("Command");
        log.debug(processBuilder.command());
        log.debug(divider);
        log.debug("Working Dir");
        log.debug(processBuilder.directory());
        log.debug(divider);
        log.debug("Environment");
        for (Map.Entry entry : processBuilder.environment().entrySet())
        {
            log.debug(entry.getKey() + ": " + entry.getValue());
        }
        log.debug(divider);
        log.debug("Redirect Error Stream?");
        log.debug(processBuilder.redirectErrorStream());
        log.debug(divider);
        log.debug("End Process Debug Information");
        log.debug(divider);
    }

    /**
     * Start the external process and setup the IO pump threads needed to
     * manage the process IO. If you call this method you must eventually call the
     * finish() method. Using this method you may execute additional code between process
     * start and finish.
     */
    public void start() {
        try {
            this.startTime = System.currentTimeMillis();
            this.process = createProcess(command, environment, workingDir);
            setupIOPumps();
        } catch (IOException e) {
            processException = new ProcessException(e);
        }
    }

    private void setupIOPumps() {
        // set up threads to feed data to and extract data from the process
        if (handler.hasInput()) {
            inputPump = new LatchedRunnable("inputPump") {
                protected void doTask() {
                    handler.provideInput(process.getOutputStream());
                }
            };
        }
        else
        {
            try
            {
                process.getOutputStream().close();
            }
            catch (IOException e)
            {
                processException = new ProcessException(e);
            }
        }

        errorPump = new LatchedRunnable("errorPump") {
            protected void doTask() {
                try {
                    handler.processError(process.getErrorStream());
                } catch (Throwable e) {
                    if (!isCancelled()) {
                        log.debug(name + ": Process wasn't cancelled, storing exception", e);
                        processException = new ProcessException(e);
                    }
                    else {
                        log.debug(name + ": Process cancelled ignoring exception", e);
                    }
                }
            }
        };

        outputPump = new LatchedRunnable("outputPump") {
            protected void doTask() {
                try {
                    handler.processOutput(process.getInputStream());
                } catch (Throwable e) {
                    if (!isCancelled()) {
                        log.debug(name + ": Process wasn't cancelled, storing exception", e);
                        processException = new ProcessException(e);
                    }
                    else {
                        log.debug(name + ": Process cancelled ignoring exception", e);
                    }
                }
            }
        };

        // tickle the dog initially
        resetWatchdog();
        handler.setWatchdog(this);

        pumpThreadPool.execute(errorPump);
        pumpThreadPool.execute(outputPump);
        if (inputPump != null) {
            pumpThreadPool.execute(inputPump);
        }
    }

    /**
     * Finish process execution. This method should be called after you have called the
     * start() method.
     */
    public void finish() {
        if (process != null) {
            try {
                do {
                    long checkTime = getTimeoutTime();
                    awaitPump(outputPump, checkTime);
                    awaitPump(inputPump, checkTime);
                    awaitPump(errorPump, checkTime);
                } while (!isTimedOut() && arePumpsRunning() && !Thread.currentThread().isInterrupted());
            } finally {

                if (Thread.currentThread().isInterrupted()) {
                    cancel();

                    // All is good, now clearing interrupted state of current thread.
                    Thread.interrupted();
                }

                int exitCode  = wrapUpProcess();
                handler.complete(exitCode, processException);
            }
        } else {
            handler.complete(-1, processException);
        }
    }
    
    /**
     * Notifies all ProcessMonitors of the 'beforeStart' event.
     */
    private void notifyBeforeStart() {
        for (ProcessMonitor monitor: monitors) {
            try {
                monitor.onBeforeStart(this);
            } catch (Exception e) {
                // catch and log error, but continue
                Logger.getLogger(ExternalProcess.class).error("Error while processing 'beforeStarted' event:", e);
            }
        }
    }

    /**
     * Notifies all ProcessMonitors of the 'afterFinished' event.
     */
    private void notifyAfterFinished() {
        for (ProcessMonitor monitor: monitors) {
            try {
                monitor.onAfterFinished(this);
            } catch (Exception e) {
                Logger.getLogger(ExternalProcess.class).error("Error while processing 'afterFinished' event:", e);
            }
        }
    }
    
    /**
     * Execute the external command. When this method returns, the process handler
     * provided at construction time should be consulted to collect exit code, exceptions,
     * process output, etc.
     */
    public void execute() {
        notifyBeforeStart();
        try {
            start();
            finish();
        } finally {
            notifyAfterFinished();
        }
    }

    /**
     * Executes the external command. While it is running, the given runnable is executed.
     * The external command is not checked until the runnable completes
     *
     * @param runnable A task to perform while the external command is running.
     */
    public void executeWhile(Runnable runnable) {
        start();
        if (runnable != null) {
            runnable.run();
        }
        finish();
    }

    public String getCommandLine() {
        StringBuilder sb = new StringBuilder();
        for (String s : command) {
            sb.append(s);
            sb.append(" ");
        }
        return sb.toString();
    }

    /**
     * Wait a given time for the process to finish
     *
     * @param maxWait the maximum amount of time in milliseconds to wait for the process to finish
     *
     * @return true if the process has finished.
     */
    public boolean finish(int maxWait) {
        if (process != null) {
            boolean finished = false;
            try {
                long endTime = System.currentTimeMillis() + maxWait;
                awaitPump(outputPump, endTime);
                awaitPump(inputPump, endTime);
                awaitPump(errorPump, endTime);
            } finally {
                if (!arePumpsRunning()) {
                    // process finished
                    finished = true;
                    int exitCode = wrapUpProcess();
                    handler.complete(exitCode, processException);
                }
            }
            return finished;
        } else {
            handler.complete(-1, processException);
            return true;
        }
    }

    private int wrapUpProcess() {
        int exitCode = -1;
        boolean processIncomplete = true;
        boolean interrupted = false;
        try {
            exitCode = process.exitValue();
            processIncomplete = false;
        } catch (IllegalThreadStateException e) {
            // process still running - could be a race to have the process finish so wait a little to be sure
            while (processIncomplete && System.currentTimeMillis() - getTimeoutTime() < 10) {
                // we are currently before the end of the period (within 10ms slack), so process probably not ready yet
                try {
                    Thread.sleep(100);
                    exitCode = process.exitValue();
                    processIncomplete = false;
                } catch (InterruptedException e1) {
                    processIncomplete = true;
                    interrupted = true;
                    break;
                } catch (IllegalThreadStateException e2) {
                    // ignore and try in the next loop
                }
            }
        } finally {
            cancel();
        }

        if (processIncomplete && !interrupted) {
            processException = new ProcessTimeoutException("process timed out");
        }
        return exitCode;
    }

    private void awaitPump(LatchedRunnable runnable, long latestTime) {
        if (runnable != null) {
            long timeout = latestTime - System.currentTimeMillis();
            if (timeout < 1) {
                timeout = 1;
            }
            runnable.await(timeout);
        }
    }

    /**
     * Cancel should be called if you wish to interrupt process execution.
     */
    public void cancel() {
        this.cancelled = true;
        if (outputPump != null) {
            outputPump.cancel();
        }

        if (inputPump != null) {
            inputPump.cancel();
        }

        if (errorPump != null) {
            errorPump.cancel();
        }

        if (process != null)
        {
           //if (isWindows())
           //{
           //    try
           //    {
           //        log.error("Killing process with winp");
           //        final WinProcess winProcess = new WinProcess(process);
           //        winProcess.killRecursively();
           //    }
           //    catch (Throwable e)
           //    {
           //        log.error("Could not recursively kill windows process using winp; falling back to java.lang.Process#destroy", e);
           //        process.destroy();
           //    }
           //}
           //else
           //{
               process.destroy();
           //}
        }
    }

    public void setTimeout(long timeout) {
        this.timeout = timeout;
    }
    
    public static void shutdown() {
        if (pumpThreadPool != null) {
            pumpThreadPool.shutdown();
        }
    }
}
