package com.atlassian.jira.usercompatibility;

import com.atlassian.crowd.embedded.api.User;
import com.atlassian.crowd.embedded.impl.IdentifierUtils;
import com.atlassian.jira.component.ComponentAccessor;
import com.atlassian.jira.user.util.UserManager;
import com.atlassian.jira.util.BuildUtilsInfo;

import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;

/**
 * Helper class containing utility methods for plugins which supports both JIRA 6.0 and older.
 * In this case, cannot use directly a new {@link ApplicationUser} interface introduced in JIRA 6.0. This class contains
 * methods which can be used to perform type-safe operations.
 */
public class UserCompatibilityHelper
{
    /**
     * Converts given userObject into {@link UserWithKey}.
     * <p/>
     * This method was introduced as an answer for different return value from {@link
     * com.atlassian.jira.issue.fields.CustomField}s containing users in JIRA 6.0. In such case, {@link
     * com.atlassian.jira.issue.fields.CustomField} returns {@link User} in JIRA < 6.0, and {@link ApplicationUser} in
     * JIRA &gt;= 6.0
     *
     * @param userObject object representing User
     * @return {@link UserWithKey} object containg unique user's key and {@link User} object itself.
     * @throws IllegalArgumentException when given userObject has unknown type
     * @throws ApplicationUserUtilAccessException when necessary class/method cannot be accessed
     */
    public static UserWithKey convertUserObject(Object userObject)
    {
        if (userObject == null)
        {
            return null;
        }

        if (userObject instanceof User)
        {
            User user = (User) userObject;
            return new DelegatingUserWithKey(user, getKeyForUser(user));
        }

        if (!isRenameUserImplemented())
        {
            throw new IllegalArgumentException("Unknown class representing user: " + userObject.getClass().getName());
        }

        try
        {
            final Class<?> appUsersClass = getApplicationUserClass();
            if (!appUsersClass.isAssignableFrom(userObject.getClass()))
            {
                throw new IllegalArgumentException("Unknown class representing user: " + userObject.getClass().getName());
            }

            Method getKeyMethod = appUsersClass.getMethod("getKey");
            final String userKey = (String) getKeyMethod.invoke(userObject);

            Method getUserMethod = appUsersClass.getMethod("getDirectoryUser");
            final User user = (User) getUserMethod.invoke(userObject);

            return new DelegatingUserWithKey(user, userKey);

        }
        catch (ClassNotFoundException e)
        {
            throw new ApplicationUserUtilAccessException(e);
        }
        catch (InvocationTargetException e)
        {
            throw new ApplicationUserUtilAccessException(e);
        }
        catch (NoSuchMethodException e)
        {
            throw new ApplicationUserUtilAccessException(e);
        }
        catch (IllegalAccessException e)
        {
            throw new ApplicationUserUtilAccessException(e);
        }
    }

    /**
     * Returns unique key, which e.g. can be stored in persistent storage, for given {@link User}.
     * <p/>
     * For plugin which supports <b>only</b> JIRA >= 6.0 it is highly recommended to use {@link
     * com.atlassian.jira.user.ApplicationUsers#getKeyFor(com.atlassian.crowd.embedded.api.User)} directly
     *
     * @param user the directory User
     * @return unique Key for given user
     * @throws ApplicationUserUtilAccessException when necessary class/method cannot be accessed
     */
    public static String getKeyForUser(User user)
    {
        if (user == null)
        {
            return null;
        }

        if (isRenameUserImplemented())
        {
            try
            {

                Class<?> appUsersClass = Class.forName("com.atlassian.jira.user.ApplicationUsers");
                Method getAppUserKeyMethod = appUsersClass.getMethod("getKeyFor", User.class);
                return (String) getAppUserKeyMethod.invoke(null, user);

            }
            catch (ClassNotFoundException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
            catch (NoSuchMethodException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
            catch (InvocationTargetException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
            catch (IllegalAccessException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
        }
        else
        {
            return IdentifierUtils.toLowerCase(user.getName());
        }
    }

    /**
     * Returns {@link User} object based on specified key.
     * <p/>
     * This method should be used as an opposite of {@link UserCompabilityHelper#getKeyForUser(com.atlassian.crowd.embedded.api.User)}
     * for finding users based on string, e.g. stored in persistent storage. For plugin which supports <b>only</b> JIRA
     * >= 6.0 it is highly recommended to use {@link UserManager#getUserByKey(String)} directly.
     * <p/>
     * <b>Warning: </b><br/> {@link com.atlassian.crowd.embedded.api.User#getName()} of returned user can be not equal
     * to given key.
     *
     * @param key User's key
     * @return {@link User} object for given key
     * @throws ApplicationUserUtilAccessException when necessary class/method cannot be accessed
     */
    public static User getUserForKey(String key)
    {
        if (key == null)
        {
            return null;
        }

        if (isRenameUserImplemented())
        {
            UserManager userManager = ComponentAccessor.getUserManager();

            try
            {
                Method getUserByKeyMethod = userManager.getClass().getMethod("getUserByKey", String.class);
                Object applicationUser = getUserByKeyMethod.invoke(userManager, key);

                if (applicationUser == null)
                { return null; }

                Method getDirectoryUser = applicationUser.getClass().getMethod("getDirectoryUser");
                return (User) getDirectoryUser.invoke(applicationUser);
            }
            catch (InvocationTargetException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
            catch (NoSuchMethodException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
            catch (IllegalAccessException e)
            {
                throw new ApplicationUserUtilAccessException(e);
            }
        }
        else
        {
            return ComponentAccessor.getUserManager().getUser(key);
        }
    }

    /**
     * Tells whether given object represents a user and can be passed to {@link #convertUserObject(Object)}.
     * <p/>
     * This method was introduced as an answer for different return value from {@link
     * com.atlassian.jira.issue.fields.CustomField}s containing users in JIRA 6.0. In such case, {@link
     * com.atlassian.jira.issue.fields.CustomField} returns {@link User} in JIRA < 6.0, and {@link ApplicationUser} in
     * JIRA &gt;= 6.0
     *
     * @param object object possibly representing a User
     * @return true if object can be passed to {@link #convertUserObject(Object)}.
     * @throws ApplicationUserUtilAccessException when necessary class/method cannot be accessed
     */
    public static boolean isUserObject(Object object)
    {
        try {
            return object instanceof User || (isRenameUserImplemented() && getApplicationUserClass().isInstance(object));
        } catch (ClassNotFoundException e) {
            throw new ApplicationUserUtilAccessException(e);
        }
    }

    private static boolean isRenameUserImplemented()
    {
        final int[] versionNumbers = ComponentAccessor.getComponent(BuildUtilsInfo.class).getVersionNumbers();
        return versionNumbers[0] >= 6;

    }

    private static Class<?> getApplicationUserClass() throws ClassNotFoundException
    {
        return Class.forName("com.atlassian.jira.user.ApplicationUser");
    }
}
