package com.atlassian.user.impl.osuser;

import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.EntityNameAlphaComparator;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.security.password.Credential;
import com.atlassian.user.util.Assert;
import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.user.ImmutableException;
import com.opensymphony.user.provider.CredentialsProvider;

import java.util.*;

/**
 * An adaptor class for {@link CredentialsProvider} and some of the higher level operations of {@link com.opensymphony.user.UserManager}
 * <p/>
 * The rule is to use the credentialsProvider and/or profileProvider (for propertySets) for most things. Store() operations
 * must be called on the entity itself.
 */
public class OSUUserManager extends OSUEntityManager implements UserManager
{
    private final OSUAccessor accessor;

    public OSUUserManager(RepositoryIdentifier repository, OSUAccessor accessor)
    {
        super(repository);
        this.accessor = accessor;
    }

    public Pager<User> getUsers()
    {
        Collection<String> userNames = getUserNamesInternal();
        SortedSet<User> atlassianUsers = new TreeSet<User>(new EntityNameAlphaComparator());

        for (String username : userNames)
        {
            OSUUser user = getWrappedOSUUser(username);
            if (user != null)
                atlassianUsers.add(user);
        }

        return new DefaultPager<User>(atlassianUsers);
    }

    public Pager<String> getUserNames() throws EntityException
    {
        return new DefaultPager<String>(getUserNamesInternal());
    }

    public User getUser(String username)
    {
        return getWrappedOSUUser(username);
    }

    private OSUUser getWrappedOSUUser(String username)
    {
        com.opensymphony.user.User opensymphonyUser = getOpenSymphonyUser(username);
        return opensymphonyUser == null ? null : new OSUUser(opensymphonyUser);
    }

    private com.opensymphony.user.User getOpenSymphonyUser(String username)
    {
        String lcUsername = username.toLowerCase();
        if (accessor.getCredentialsProvider().handles(lcUsername))
            return new com.opensymphony.user.User(username.toLowerCase(), accessor);
        else
            return null;
    }

    public User createUser(String username) throws com.atlassian.user.EntityException
    {
        validateNewUserName(username);

        if (accessor.getCredentialsProvider().create(username))
        {
            accessor.getProfileProvider().create(username);
            return getWrappedOSUUser(username);
        }

        //this is a very poor condition to run into but what can be done about it?
        throw new EntityException("Was unable to create user [" + username + "] but the credentials provider ["
            + accessor.getCredentialsProvider().toString() + "] did not say why.");
    }

    public User createUser(User userTemplate, Credential credential) throws EntityException, UnsupportedOperationException
    {
        String username = userTemplate.getName();
        validateNewUserName(username);
        if (credential.isEncrypted() && credential != Credential.NONE)
            throw new IllegalArgumentException("OSUser passwords must not be encrypted");

        boolean created = accessor.getCredentialsProvider().create(username);
        if (!created)
            throw new RepositoryException("Couldn't create user [" + username + "] in credentials provider");

        accessor.getProfileProvider().create(username);
        com.opensymphony.user.User user = new com.opensymphony.user.User(username.toLowerCase(), accessor);
        user.setFullName(userTemplate.getFullName());
        user.setEmail(userTemplate.getEmail());
        try
        {
            if (credential != Credential.NONE && credential.getValue() != null)
                user.setPassword(credential.getValue());
            user.store();
        }
        catch (ImmutableException e)
        {
            throw new RepositoryException(e);
        }

        return new OSUUser(user);
    }

    private void validateNewUserName(String username) throws com.atlassian.user.impl.DuplicateEntityException
    {
        boolean usernameHandled = accessor.getCredentialsProvider().handles(username);

        if (usernameHandled)
            throw new com.atlassian.user.impl.DuplicateEntityException("User [" + username
                + "] already exists in credentialsProvider [" + accessor.getCredentialsProvider().toString() + "]");
    }

    /**
     * Encrypts the plain password, sets it on the user, and saves the user.
     */
    public void alterPassword(User user, String plainTextPass) throws EntityException
    {
        if (!(user instanceof OSUUser))
            throw new EntityException("Unsupported user type: " + user);
        
        OSUUser osUser = (OSUUser) user;
        osUser.setPassword(plainTextPass);
        saveUser(osUser);
    }

    public void removeUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        Assert.isInstanceOf(OSUUser.class, user);
        Assert.isTrue(getUser(user.getName()) != null,
            "User is not managed by this user manager: [" + user.getName() + "]");

        String userName = user.getName();

        // need to remove properties from a user as its deleted
        // See: http://jira.atlassian.com/browse/USER-192
        PropertySet propertySet = accessor.getProfileProvider().getPropertySet(userName);

        for (Object o : propertySet.getKeys())
        {
            String key = (String) o;
            propertySet.remove(key);
        }

        List<String> groupsOfUser = new ArrayList<String>(getGroupsContainingUserInternal(userName));

        for (String groupName : groupsOfUser)
        {
            accessor.getAccessProvider().removeFromGroup(userName, groupName);
        }

        boolean result = accessor.getCredentialsProvider().remove(userName);

        if (!result)
            throw new EntityException("Could not remove user!");

        accessor.getProfileProvider().remove(userName);
    }

    public boolean isReadOnly(User user)
    {
        return getUser(user.getName()) == null;
    }

    public void saveUser(User user) throws EntityException
    {
        Assert.notNull(user, "User must not be null");
        com.opensymphony.user.User osUser = getOpenSymphonyUser(user.getName());
        osUser.setFullName(user.getFullName());
        osUser.setEmail(user.getEmail());

        try
        {
            osUser.store();
        }
        catch (ImmutableException e)
        {
            throw new RepositoryException(e);
        }
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        return getUser(entity.getName()) == null ? null : repository;
    }

    public OSUAccessor getAccessor()
    {
        return accessor;
    }

    @SuppressWarnings({"unchecked"})
    private List<String> getUserNamesInternal()
    {
        return accessor.getCredentialsProvider().list();
    }

    @SuppressWarnings({"unchecked"})
    private List<String> getGroupsContainingUserInternal(String userName)
    {
        return (List<String>) accessor.getAccessProvider().listGroupsContainingUser(userName);
    }
}
