package com.atlassian.user.impl.ldap.search.page;

import com.atlassian.user.EntityException;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.repository.LdapContextFactory;
import com.atlassian.user.impl.ldap.search.LDAPPagerInfo;
import com.atlassian.user.util.LDAPUtils;
import com.atlassian.util.profiling.UtilTimerStack;
import org.apache.log4j.Logger;

import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.*;
import java.util.List;

public class LDAPMembershipToUsernamePager extends LDAPSingleStringPager
{
    private static final Logger log = Logger.getLogger(LDAPMembershipToUsernamePager.class);

    public LDAPMembershipToUsernamePager(LdapSearchProperties searchProperties, LdapContextFactory repository, LDAPPagerInfo info)
    {
        super(searchProperties, repository, info);
    }

    protected List<String> preloadSearchResult(SearchResult result, List<String> prefetched) throws EntityException
    {
        DirContext ctx = null;
        try
        {
            Attributes entityAttributes = result.getAttributes();
            String attributeToFind = returningAttributes[0];
            Attribute attr = entityAttributes.get(attributeToFind);

            if (attr != null)
            {
                NamingEnumeration interiorList = attr.getAll();
                ctx = repository.getLDAPContext();
                while (interiorList.hasMoreElements())
                    addIfFoundUser(prefetched, (String) interiorList.nextElement(), ctx);
            }
        }
        catch (Throwable t)
        {
            log.error("Error converting search result: " + result + " into list of members as usernames.", t);
        }
        finally
        {
            closeContext(ctx);
        }

        return prefetched;
    }

    /**
     * If the dn resolves to a username (by {@link #findByDN}) add it to the prefetched list
     * otherwise log it and ignore.
     *
     * @param ctx an already initialised context which can be used to look up the dn
     */
    private void addIfFoundUser(List<String> prefetched, String dn, DirContext ctx)
    {
        try
        {
            String username = findByDN(dn, ctx);
            if (username != null)
                prefetched.add(username);
        }
        catch (RepositoryException e)
        {
            log.error("Error resolving dn [ " + dn + " ] to a username", e);
        }
    }

    /**
     * @param ctx an already initialised context which can be used to look up the dn
     */
    private String findByDN(String dn, DirContext ctx) throws RepositoryException
    {
        String usernameAttribure = searchProperties.getUsernameAttribute();
        SearchControls ctls = LDAPUtils.createSearchControls(
                new String[]{usernameAttribure}, true, searchProperties.getTimeLimitMillis());

        try
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.push(this.getClass().getName() + "_search_JNDI_RAW_" + searchProperties.getUserFilter());

            NamingEnumeration<SearchResult> userSearchEnum = ctx.search(dn, searchProperties.getUserFilter(), ctls);
            if (userSearchEnum.hasMoreElements())
            {
                SearchResult sr = userSearchEnum.next();
                if (sr != null && sr.getAttributes() != null && sr.getAttributes().get(usernameAttribure) != null)
                {
                    String username = (String) sr.getAttributes().get(usernameAttribure).get();
                    if (log.isDebugEnabled())
                        log.debug("LDAPMembershipToUse.findByDN [ " + dn + " ] username [ " + username + " ]");
                    return username;
                }
            }
            return null;
        }

        catch (NamingException e)
        {
            throw new RepositoryException(e);
        }
        finally
        {
            if (UtilTimerStack.isActive())
                UtilTimerStack.pop(this.getClass().getName() + "_search_JNDI_RAW_" + searchProperties.getUserFilter());
        }

    }

    private void closeContext(DirContext ctx)
    {
        try
        {
            if (ctx != null)
                ctx.close();
        }
        catch (NamingException e)
        {
            log.warn("Exception closing context", e);
        }
    }
}
