package com.atlassian.user.impl.ldap.configuration;

import com.atlassian.user.configuration.*;
import com.atlassian.user.impl.ldap.properties.factory.LdapConnectionPropertiesFactory;
import com.atlassian.user.impl.ldap.properties.factory.LdapMembershipPropertiesFactory;
import com.atlassian.user.impl.ldap.properties.factory.LdapSearchPropertiesFactory;
import com.atlassian.user.impl.ldap.properties.LdapConnectionProperties;
import com.atlassian.user.impl.ldap.properties.LdapSearchProperties;
import com.atlassian.user.impl.ldap.properties.LdapMembershipProperties;
import com.atlassian.user.impl.ldap.repository.DefaultLdapContextFactory;
import com.atlassian.user.impl.ldap.search.DefaultLdapFilterFactory;

import java.util.HashMap;
import java.util.Map;
import java.util.Properties;

public class LdapRepositoryProcessor extends DefaultRepositoryProcessor
{
    public RepositoryAccessor process(RepositoryConfiguration config) throws ConfigurationException
    {
        setConnectionPoolProperties(config);

        Properties schemaMappingsProperties = parseSchemaMappings(config);
        Properties serverProperties = parseLDAPServerConfiguration(config);

        LdapConnectionProperties connectionProperties = new LdapConnectionPropertiesFactory().createInstance(serverProperties);
        LdapSearchProperties searchProperties = new LdapSearchPropertiesFactory().createInstance(schemaMappingsProperties);
        LdapMembershipProperties membershipProperties = new LdapMembershipPropertiesFactory().createInstance(schemaMappingsProperties);

        config.addComponent("connectionProperties", connectionProperties);
        config.addComponent("searchProperties", searchProperties);
        config.addComponent("membershipProperties", membershipProperties);

        config.addComponent(Configuration.LDAP_SCHEMA_MAPPINGS_PROPERTIES, schemaMappingsProperties);
        config.addComponent(Configuration.SERVER, serverProperties);

        config.addComponent("contextFactory", new DefaultLdapContextFactory(connectionProperties));
        config.addComponent("filterFactory", new DefaultLdapFilterFactory(searchProperties));

        config.addComponent(Configuration.USERADAPTOR, createBean(Configuration.USERADAPTOR, config));
        config.addComponent(Configuration.USERFACTORY, createBean(Configuration.USERFACTORY, config));
        config.addComponent(Configuration.GROUPFACTORY, createBean(Configuration.GROUPFACTORY, config));
        config.addComponent(Configuration.GROUPADAPTOR, createBean(Configuration.GROUPADAPTOR, config));

        return super.process(config);
    }

    public void setConnectionPoolProperties(RepositoryConfiguration config)
    {
        Map<String, String> properties = new HashMap<String, String>();
        properties.put("com.sun.jndi.ldap.connect.pool.maxsize", config.getStringComponent(Configuration.MAXSIZE));
        properties.put("com.sun.jndi.ldap.connect.pool.initsize", config.getStringComponent(Configuration.INITSIZE));
        properties.put("com.sun.jndi.ldap.connect.pool.prefsize", config.getStringComponent(Configuration.PREFSIZE));
        properties.put("com.sun.jndi.ldap.connect.pool.debug", config.getStringComponent(Configuration.DEBUG));
        properties.put("com.sun.jndi.ldap.connect.pool.protocol", config.getStringComponent(Configuration.SECURITY_PROTOCOL));
        properties.put("com.sun.jndi.ldap.connect.pool.authentication", config.getStringComponent(Configuration.POOL_AUTHENTICATION));
        properties.put("com.sun.jndi.ldap.connect.pool.timeout", config.getStringComponent(Configuration.TIMEOUT));
        properties.put("javax.net.ssl.trustStore", config.getStringComponent(Configuration.TRUSTSTORE));

        for (Map.Entry<String, String> entry : properties.entrySet())
        {
            String value = entry.getValue();
            if (value != null) System.setProperty(entry.getKey(), value);
        }
    }

    public Properties parseLDAPServerConfiguration(RepositoryConfiguration config)
    {
        Properties serverProperties = new Properties();

        String[] propertyKeys = new String[]{
            Configuration.HOST,
            Configuration.PORT,
            Configuration.SECURITY_PRINCIPAL,
            Configuration.SECURITY_CREDENTIAL,
            Configuration.SECURITY_PROTOCOL,
            Configuration.AUTHENTICATION,
            Configuration.BASE_CONTEXT,
            Configuration.BATCH_SIZE,
            Configuration.INITIAL_CONTEXT_FACTORY_JNDI,
            Configuration.POOLING_ON,
            Configuration.CONNECT_TIMEOUT,
            Configuration.READ_TIMEOUT,
        };

        for (String key : propertyKeys)
        {
            String value = (String) config.getComponent(key);
            if (value == null) continue;
            serverProperties.put(key, value);
        }
        return serverProperties;
    }

    public Properties parseSchemaMappings(RepositoryConfiguration config)
    {
        Properties schemaMappingsProperties = new Properties();

        String[] propertyKeys = new String[]{
            Configuration.BASE_USER_NAMESPACE,
            Configuration.BASE_GROUP_NAMESPACE,
            Configuration.USERNAME_ATTRIBUTE,
            Configuration.GROUPNAME_ATTRIBUTE,
            Configuration.USER_SEARCH_FILTER,
            Configuration.GROUP_SEARCH_FILTER,
            Configuration.FIRSTNAME_ATTRIBUTE,
            Configuration.SURNAME_ATTRIBUTE,
            Configuration.EMAIL_ATTRIBUTE,
            Configuration.MEMBERSHIP_ATTRIBUTE,
            Configuration.USER_SEARCH_ALL_DEPTHS,
            Configuration.GROUP_SEARCH_ALL_DEPTHS,
            Configuration.USE_UNQUALIFIED_USER_NAME_FOR_MEMBERSHIP_COMPARISON,
            Configuration.TIME_TO_LIVE,
        };

        for (String key : propertyKeys)
        {
            String value = (String) config.getComponent(key);
            if (value == null) continue;
            schemaMappingsProperties.setProperty(key, value);
        }

        return schemaMappingsProperties;
    }
}
