package com.atlassian.user.impl.cache;

import com.atlassian.cache.CacheFactory;
import com.atlassian.user.Entity;
import com.atlassian.user.EntityException;
import com.atlassian.user.Group;
import com.atlassian.user.GroupManager;
import com.atlassian.user.User;
import com.atlassian.user.repository.RepositoryIdentifier;
import com.atlassian.user.search.page.DefaultPager;
import com.atlassian.user.search.page.Pager;
import com.atlassian.user.search.page.PagerUtils;
import org.apache.log4j.Category;

import java.util.LinkedList;
import java.util.List;

/**
 * Uses a generic caching strategy to provide caching for any implementation of
 * {@link com.atlassian.user.GroupManager}.
 * <p/>
 * Typically used by putting the 'caching="true"' attribute on the repository in
 * the XML configuration. The default repository loader will then wrap the
 * GroupManager implementation in an instance of this class.
 * <p/>
 * To minimise memory use, except for the direct name->object caches, the
 * underlying caches do not store the actual group or user objects. Instead,
 * they store just the user or group name and methods in this class retrieve the
 * objects as required. The name->object caches will ensure that is relatively
 * efficient.
 */
public class CachingGroupManager implements GroupManager
{
    private static final Category log = Category.getInstance(CachingGroupManager.class);

    protected final GroupManager underlyingGroupManager;
    protected final CacheFactory cacheFactory;

    protected GroupCache groupCache = null;
    protected MembershipCache membershipCache = null;
    protected GroupsForUserCache groupsForUserCache = null;
    protected EntityRepositoryCache entityRepositoryCache = null;

    public CachingGroupManager(GroupManager underlyingGroupManager, CacheFactory cacheFactory)
    {
        this.underlyingGroupManager = underlyingGroupManager;
        this.cacheFactory = cacheFactory;
        initialiseCaches();
    }

    /**
     * Caches the list of groups retrieved for a particular user. This incurs a performance hit
     * the first time a user's groups are retrieved, because we iterate over the entire pager
     * and cache all groups to which the user belongs.
     *
     * The cached stores only group names to reduce memory usage. Since this method must return
     * {@link com.atlassian.user.Group} objects in the Pager, they are retrieved using {@link #getGroup(String)}. This
     * should be relatively fast for subsequent lookups because this method also uses a cache.
     *
     * @param user the user whose groups will be retrieved. Must not be null.
     * @return a Pager containing all the groups in the underlying group manager which have this user
     * as a member. Each item in the list is an instance of the {@link com.atlassian.user.Group} type used by the underlying manager.
     * Returns an empty pager if no such groups exist.
     * @throws com.atlassian.user.EntityException if there is an error retrieving the groups.
     */
    public Pager<Group> getGroups(User user) throws EntityException
    {
        if (user == null)
            throw new IllegalArgumentException("User cannot be null.");
        if (log.isInfoEnabled())
            log.info("Retrieving groups for user [" + user.getName() + "]");

        List<String> groupNames = groupsForUserCache.get(user);

        if (groupNames != null)
        {
            if (log.isDebugEnabled())
                log.debug("Cache hit. Returning pager with " + groupNames.size() + " items.");

            List<Group> groups = new LinkedList<Group>();

            for (Object groupName1 : groupNames)
            {
                groups.add(getGroup((String) groupName1));
            }
            return new DefaultPager<Group>(groups);
        }

        if (log.isDebugEnabled())
            log.debug("Cache miss. Retrieving groups from underlying group manager.");

        List<Group> groups = new LinkedList<Group>();
        groupNames = new LinkedList<String>();

        for (Group group : underlyingGroupManager.getGroups(user))
        {
            groups.add(group);
            groupCache.put(group.getName(), group);
            groupNames.add(group.getName());
        }

        if (log.isDebugEnabled())
            log.debug("Retrieved " + groupNames.size() + " groups for user [" + user + "], putting in cache.");

        groupsForUserCache.put(user, groupNames);

        return new DefaultPager<Group>(groups);
    }

    public List<Group> getWritableGroups()
    {
        return underlyingGroupManager.getWritableGroups();
    }

    public Group getGroup(String groupName) throws EntityException
    {
        Group cachedGroup = groupCache.get(groupName);
        if (cachedGroup != null)
        {
            return GroupCache.NULL_GROUP.equals(cachedGroup) ? null : cachedGroup;
        }
        else
        {
            Group group = underlyingGroupManager.getGroup(groupName);
            groupCache.put(groupName, group);
            return group;
        }
    }

    public Group createGroup(String groupName) throws EntityException
    {
        Group createdGroup = underlyingGroupManager.createGroup(groupName);

        if (createdGroup != null)
            groupCache.put(createdGroup.getName(), createdGroup);

        return createdGroup;
    }

    public void removeGroup(Group group) throws EntityException
    {
        List<String> memberNames = PagerUtils.toList(getMemberNames(group));
        underlyingGroupManager.removeGroup(group);
        groupCache.remove(group.getName());
        groupsForUserCache.remove(memberNames);
        membershipCache.remove(memberNames, group);
        entityRepositoryCache.remove(group);
    }

    public void addMembership(Group group, User user) throws EntityException
    {
        underlyingGroupManager.addMembership(group, user);
        membershipCache.put(user, group, true);
        groupsForUserCache.remove(user);
    }

    public boolean hasMembership(Group group, User user) throws EntityException
    {
        if(group==null)
            return false;
        
        Boolean membershipCheckElement = membershipCache.get(user, group);

        if (membershipCheckElement != null)
        {
            return membershipCheckElement;
        }
        else
        {
            boolean isMember = underlyingGroupManager.hasMembership(group, user);
            membershipCache.put(user, group, isMember);
            return isMember;
        }
    }

    public void removeMembership(Group group, User user) throws EntityException
    {
        underlyingGroupManager.removeMembership(group, user);
        membershipCache.remove(user, group);
        groupsForUserCache.remove(user);
    }

    public RepositoryIdentifier getRepository(Entity entity) throws EntityException
    {
        RepositoryIdentifier cachedRepository = entityRepositoryCache.get(entity);
        if (cachedRepository != null)
            return cachedRepository;

        RepositoryIdentifier repository = underlyingGroupManager.getRepository(entity);
        entityRepositoryCache.put(entity, repository);
        return repository;
    }

    public Pager<Group> getGroups() throws EntityException
    {
        return underlyingGroupManager.getGroups();
    }

    public Pager<String> getMemberNames(Group group) throws EntityException
    {
        return underlyingGroupManager.getMemberNames(group);
    }

    public Pager<String> getLocalMemberNames(Group group) throws EntityException
    {
        return underlyingGroupManager.getLocalMemberNames(group);
    }

    public Pager<String> getExternalMemberNames(Group group) throws EntityException
    {
        return underlyingGroupManager.getExternalMemberNames(group);
    }

    public boolean supportsExternalMembership() throws EntityException
    {
        return underlyingGroupManager.supportsExternalMembership();
    }

    public boolean isReadOnly(Group group) throws EntityException
    {
        return underlyingGroupManager.isReadOnly(group);
    }

    public RepositoryIdentifier getIdentifier()
    {
        return underlyingGroupManager.getIdentifier();
    }

    public boolean isCreative()
    {
        return underlyingGroupManager.isCreative();
    }

    /**
     * To be called once the underlyingGroupManager and cacheManager have been set up by a constructor.
     */
    private void initialiseCaches()
    {
        entityRepositoryCache = new EntityRepositoryCache(cacheFactory, getCacheKey("repositories"));
        groupCache = new GroupCache(cacheFactory, getCacheKey("groups"));
        membershipCache = new MembershipCache(cacheFactory, getCacheKey("groups_hasMembership"));
        groupsForUserCache = new GroupsForUserCache(cacheFactory, getCacheKey("groups_getGroupsForUser"));
    }

    /**
     * Generates a unique cache key. This cache key should not be shared by other instances of CachingGroupManager that delegate to different underlying group managers.
     */
    private String getCacheKey(String cacheName)
    {
        String className = underlyingGroupManager.getClass().getName();
        String repositoryKey = underlyingGroupManager.getIdentifier().getKey(); // use the repository key to compose the cache key, so we can have a cache per (repository + groupManager) combination.
        return className + "." + repositoryKey + "." + cacheName;
    }
}
