package com.atlassian.user.util.migration;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.sql.DataSource;

import com.atlassian.user.security.password.Credential;
import net.sf.hibernate.HibernateException;
import net.sf.hibernate.Session;
import net.sf.hibernate.SessionFactory;

import org.apache.log4j.Logger;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.RowCallbackHandler;
import org.springframework.jdbc.datasource.SingleConnectionDataSource;
import org.springframework.orm.hibernate.SessionFactoryUtils;

import com.atlassian.user.EntityException;
import com.atlassian.user.ExternalEntity;
import com.atlassian.user.Group;
import com.atlassian.user.GroupManager;
import com.atlassian.user.User;
import com.atlassian.user.UserManager;
import com.atlassian.user.configuration.DefaultDelegationAccessor;
import com.atlassian.user.configuration.DelegationAccessor;
import com.atlassian.user.configuration.RepositoryAccessor;
import com.atlassian.user.impl.DefaultUser;
import com.atlassian.user.impl.RepositoryException;
import com.atlassian.user.impl.hibernate.DefaultExternalEntityDAO;
import com.atlassian.user.impl.hibernate.DefaultHibernateUser;
import com.atlassian.user.impl.hibernate.properties.HibernatePropertySetFactory;
import com.atlassian.user.impl.osuser.OSUAccessor;
import com.atlassian.user.impl.osuser.OSUUserManager;
import com.opensymphony.user.provider.AccessProvider;

/**
 * <p>
 * Makes a raw JDBC connection to os_user tables and copies across information into the supplied
 * {@link com.atlassian.user.UserManager}, {@link com.atlassian.user.GroupManager}, and
 * {@link com.atlassian.user.properties.PropertySetFactory}
 * </p>
 * <p>
 * <strong>Note</strong> that this class is not safe for multi-threaded use since it maintains reference to the
 * HibernateSession it is running within for it's duration.
 * </p>
 */

public class OSUEntityMigrator implements EntityMigrator
{
    private static final Logger log = Logger.getLogger(OSUEntityMigrator.class);

    private static final String OSUSER_REPOSITORY_KEY = "osuserRepository";

    private UserManager targetUserManager;
    private GroupManager targetGroupManager;

    private AccessProvider osAccessProvider;

    private final SessionFactory sessionFactory;

    private final DefaultExternalEntityDAO externalEntityDAO;
    
    /**
     * The session that will be used for run of this migrator. A reference is maintained to ensure it doesn't get
     * garbage collected (and the connection closed) during the run.
     */
    private Session hibernateSession;

    public OSUEntityMigrator(RepositoryAccessor osuserRepositoryAccessor, RepositoryAccessor repositoryAccessor, SessionFactory sessionFactory)
    {
        if (osuserRepositoryAccessor == null)
            throw new IllegalArgumentException("osuserRepositoryAccessor is required.");
        if (repositoryAccessor == null)
            throw new IllegalArgumentException("targetRepositoryAccessor is required.");
        if (sessionFactory == null)
            throw new IllegalArgumentException("sessionFactory is required.");

        this.sessionFactory = sessionFactory;
        this.externalEntityDAO = new DefaultExternalEntityDAO(sessionFactory);

        final DelegationAccessor targetRepositoryAccessor = getNonOSUserRepositoryAccessor(repositoryAccessor);
        if (!targetRepositoryAccessor.getRepositoryAccessors().isEmpty())
        {
            final UserManager osUserManager = osuserRepositoryAccessor.getUserManager();
            if (osUserManager == null)
                throw new IllegalArgumentException("osUserManager is required.");

            final OSUAccessor osuAccessor = ((OSUUserManager) osUserManager).getAccessor();
            if (osuAccessor == null)
                throw new IllegalArgumentException("osuAccessor is required.");

            osAccessProvider = osuAccessor.getAccessProvider();
            if (osAccessProvider == null)
                throw new IllegalArgumentException("osAccessProvider is required.");

            targetUserManager = targetRepositoryAccessor.getUserManager();
            targetGroupManager = targetRepositoryAccessor.getGroupManager();

            if (targetUserManager == null)
                throw new IllegalArgumentException("userManager is required.");
            if (targetGroupManager == null)
                throw new IllegalArgumentException("groupManager is required.");
        }
    }

    private DelegationAccessor getNonOSUserRepositoryAccessor(RepositoryAccessor repositoryAccessor)
    {
        final DelegationAccessor nonOSUserDelegationAccessor = new DefaultDelegationAccessor();
        if (repositoryAccessor instanceof DelegationAccessor)
        {
            final DelegationAccessor delegationAccessor = (DelegationAccessor) repositoryAccessor;
            for (Iterator iterator = delegationAccessor.getRepositoryAccessors().iterator(); iterator.hasNext();)
            {
                final RepositoryAccessor accessor = (RepositoryAccessor) iterator.next();
                if (!OSUSER_REPOSITORY_KEY.equals(accessor.getIdentifier().getKey()))
                    nonOSUserDelegationAccessor.addRepositoryAccessor(accessor);
            }
            return nonOSUserDelegationAccessor;
        }
        else
        {
            if (!OSUSER_REPOSITORY_KEY.equals(repositoryAccessor.getIdentifier().getKey()))
                nonOSUserDelegationAccessor.addRepositoryAccessor(repositoryAccessor);
        }
        return nonOSUserDelegationAccessor;
    }

    public boolean hasExistingUsers()
    {
        try
        {
            return !targetUserManager.getUsers().isEmpty();
        }
        catch (EntityException e)
        {
            throw new RuntimeException(e);
        }
    }

    /**
     * The method is organised in a 'strange' way for performance reasons. DON'T change it. Every white space is there for a reason :)
     * The performace problem was: every time we add a member to a hibernate group, hibernate marks this group object as dirty.
     * Because hibernate group contains the list of all its members and needs to iterate through all of them when flush() is called.
     * flush() is called every time hibernate thinks it needs to do it, for example when we call getUser() it first calls flush to make
     * sure we get up to date data. We structured the code in a way so that flush is not called until we add all users to a group.
     * That's why we cache the list of users. We also rely on the fact that targetGroupManager maintains its own cache and does not need
     * to call flush() each time we get a group by a groupname.
     *
     * @throws RepositoryException if there is no non OSUser repository was included in target repository Accessor
     */
    public void migrate(MigratorConfiguration config, MigrationProgressListener progressListener) throws EntityException
    {
        if (targetUserManager == null)
        {
            throw new RepositoryException("No non OSUser repository configured. Cannot perform migration.");
        }
        
        hibernateSession = SessionFactoryUtils.getSession(sessionFactory, true);        

        OSUserDao osUserDao = new OSUserDao(getDataSource());
        final Map<Long, DefaultUser> users = osUserDao.findAllUsers();
        final Map<String, List<String>> userGroups = osUserDao.findAllUserGroups(users);
        Map<User, Boolean> migratedUsers = migrateUsers(progressListener, users);

        // migrate group memberships
        for (Map.Entry<User, Boolean> userEntry : migratedUsers.entrySet())
        {
            final User user = userEntry.getKey();
            migrateUserGroupMembership(user, userGroups.get(user.getName()), userEntry.getValue(), config, progressListener);
        }

        // some groups could be empty hence they would not be migrated
        // when migrating users' membership -- need to migrate them explicitly
        migrateGroups(progressListener);
    }

    private Map<User, Boolean> migrateUsers(MigrationProgressListener progressListener, Map<Long, DefaultUser> users)
            throws EntityException
    {
        // starting user migration
        progressListener.userMigrationStarted(users.size());

        Map<User, Boolean> migratedUsers = new HashMap<User, Boolean>();

        int i = 0;
        for (Iterator<Map.Entry<Long,DefaultUser>> it = users.entrySet().iterator(); it.hasNext(); i++)
        {
            Map.Entry<Long,DefaultUser> userEntry = it.next();
            final Long osUserId = userEntry.getKey();
            final User user = userEntry.getValue();

            User existingUser = targetUserManager.getUser(user.getName());
            if(existingUser == null)
            {
                User newUser = addUser(targetUserManager, (DefaultUser) user);
                migratedUsers.put(newUser, Boolean.TRUE);
            }
            else
            {
                migratedUsers.put(existingUser, Boolean.FALSE);
            }
            migratePropertySet(osUserId, user);


            progressListener.userMigrated();
            if (i % 100 == 0){
                try
                {
                    hibernateSession.flush();
                    hibernateSession.clear();
                }
                catch (HibernateException e)
                {
                    log.error(e);
                }
            }
        }
        // users migrated, starting group migration
        progressListener.userMigrationComplete();
        return migratedUsers;
    }

    private void migrateGroups(MigrationProgressListener progressListener)
            throws EntityException
    {
        // noinspection unchecked
        List<String> groupNames = osAccessProvider.list();
        progressListener.groupMigrationStarted(groupNames.size());

        for (String groupName : groupNames)
        {
            getOrCreateGroup(groupName);
            progressListener.groupMigrated();
        }
        // group migration complete
        progressListener.groupMigrationComplete();
    }

    private void migrateUserGroupMembership(User user, List<String> userGroups, boolean isCreatedUser,
        MigratorConfiguration config, MigrationProgressListener progressListener) throws EntityException
    {
        if (userGroups != null)
        {
            for (String groupName : userGroups)
            {
                Group group = getOrCreateGroup(groupName);
                if (isCreatedUser || config.isMigrateMembershipsForExistingUsers())
                {
                    if (log.isInfoEnabled())
                        log.info("Adding member <" + user.getName() + "> to group <" + groupName + ">");
                    if (!targetGroupManager.isReadOnly(group))
                    {
                        targetGroupManager.addMembership(group, user);
                    }
                    else
                    {
                        progressListener.readonlyGroupMembershipNotMigrated(group.getName(), user.getName());
                    }
                }
            }
        }
    }

    private void migratePropertySet(Long userId, User user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Migrating properties for <" + user.getName() + ">");

        final User targetUser = targetUserManager.getUser(user.getName());
        final String entityName = getEntityName(targetUser);
        final long entityId = getEntityId(targetUser);

        final JdbcTemplate template = new JdbcTemplate(getDataSource());

        if (template.queryForInt("SELECT count(*) FROM OS_PROPERTYENTRY WHERE entity_name=? AND entity_id=?", new Object[] {entityName, entityId}) == 0)
        {
            template.query("SELECT * FROM OS_PROPERTYENTRY WHERE entity_name = 'OSUser_user' AND entity_id = ? AND entity_key <> 'fullName' AND entity_key <> 'email'", new Object[]{userId}, new RowCallbackHandler()
            {
                public void processRow(ResultSet resultSet) throws SQLException
                {
                    template.update("INSERT INTO OS_PROPERTYENTRY (entity_name,entity_id,entity_key,key_type,boolean_val,double_val,string_val,long_val,int_val,date_val) VALUES (?,?,?,?,?,?,?,?,?,?)", new Object[]{
                        entityName,
                        entityId,
                        resultSet.getString("entity_key"),
                        resultSet.getInt("key_type"),
                        resultSet.getBoolean("boolean_val"),
                        resultSet.getDouble("double_val"),
                        resultSet.getString("string_val"),
                        resultSet.getLong("long_val"),
                        resultSet.getInt("int_val"),
                        resultSet.getTimestamp("date_val")
                    });
                }
            });
        }
    }

    private String getEntityName(User user)
    {
        if (isExternalUser(user))
        {
            return HibernatePropertySetFactory.EXTERNAL_ENTITY + "_" + user.getName();
        }
        return HibernatePropertySetFactory.LOCAL_USER + "_" + user.getName();
    }

    private long getEntityId(User user)
    {
        if (!isExternalUser(user))
        {
            return ((DefaultHibernateUser) user).getId();
        }
        ExternalEntity externalEntity = externalEntityDAO.getExternalEntity(user.getName());
        if (externalEntity != null)
        {
            return externalEntity.getId();
        }
        return externalEntityDAO.createExternalEntity(user.getName()).getId();
    }

    private boolean isExternalUser(User user)
    {
        return !(user instanceof DefaultHibernateUser);
    }

    /**
     * Adds the given user using the user manager.
     *
     * @param userManager the user manager to add the user to
     * @param user the user to add
     * @throws EntityException if a problem occur adding the user in the user manager
     */
    private User addUser(UserManager userManager, DefaultUser user) throws EntityException
    {
        if (log.isInfoEnabled()) log.info("Adding user <" + user.getName() + ">");
        Credential credential = user.getPassword() == null ?
            Credential.NONE :
            Credential.encrypted(user.getPassword());
        if (user.getFullName() == null) user.setFullName("");
        if (user.getEmail() == null) user.setEmail("");
        return userManager.createUser(user, credential);
    }

    private Group getOrCreateGroup(String groupName) throws EntityException
    {
        Group group = targetGroupManager.getGroup(groupName);
        if (group == null)
        {
            if (log.isInfoEnabled()) log.info("Creating group <" + groupName + ">");
            group = targetGroupManager.createGroup(groupName);
        }

        return group;
    }

    private DataSource getDataSource()
    {
        Connection conn;
        try
        {
            conn = hibernateSession.connection();
        }
        catch (HibernateException e)
        {
            throw new RuntimeException(e);
        }
        return new SingleConnectionDataSource(conn, true);
    }
}
