package com.atlassian.user.search.query;

import com.atlassian.user.util.Assert;

import java.util.Collection;
import java.util.Collections;

/**
 * Checks that a boolean query contains consistent terms.
 */
public final class QueryValidator
{
    /**
     * Checks for validation errors using {@link #validateQuery(Query)} and throws the first
     * problem as an exception if any are found.
     *
     * @param query the query to validate
     * @throws EntityQueryException if there is a validation error found in the query
     */
    public void assertValid(Query query) throws EntityQueryException
    {
        Collection<EntityQueryException> validationErrors = validateQuery(query);
        if (!validationErrors.isEmpty())
            throw validationErrors.iterator().next();
    }

    /**
     * Checks that a query can be handled by implementations of {@link EntityQueryParser}.
     *
     * @param query the query to validate
     * @return a collection of {@link EntityQueryException}s or an empty collection if there are no problems with the query
     */
    public Collection<EntityQueryException> validateQuery(Query query)
    {
        Assert.notNull(query, "Query should not be null");
        if (!(query instanceof BooleanQuery))
            return Collections.emptyList();

        BooleanQuery booleanQuery = (BooleanQuery) query;
        Class<? extends Query> queryType = getNestedQueryType(booleanQuery);
        return validateBooleanQuery(booleanQuery, queryType);
    }

    /**
     * Checks that all terms in a {@link BooleanQuery} match the same criteria.
     *
     * @param booleanQuery the query to validate
     * @return a collection of {@link EntityQueryException}s, or empty collection if there were no validation errors
     */
    public Collection<EntityQueryException> validateBooleanQuery(BooleanQuery<?> booleanQuery, Class<? extends Query> queryType)
    {
        for (Query query : booleanQuery.getQueries())
        {
            if (query instanceof BooleanQuery)
            {
                return validateBooleanQuery((BooleanQuery) query, queryType);
            }
            if (!queryType.isInstance(query))
            {
                //noinspection ThrowableInstanceNeverThrown
                return Collections.singleton(new EntityQueryException("Boolean query type " + queryType.getName() +
                    " isn't matched by clause: " + query));
            }
        }
        return Collections.emptyList();
    }

    /**
     * Returns the type of the first clause in the provided BooleanQuery that is not itself a BooleanQuery.
     *
     * @throws IllegalArgumentException if the query contains a term which is not one of the primitive query types: UserQuery or GroupQuery
     */
    private Class<? extends Query> getNestedQueryType(BooleanQuery<?> booleanQuery) throws IllegalArgumentException
    {
        Query query = booleanQuery.getQueries().get(0);
        if (query instanceof BooleanQuery)
        {
            return getNestedQueryType((BooleanQuery) query);
        }
        if (UserQuery.class.isInstance(query))
            return UserQuery.class;
        if (GroupQuery.class.isInstance(query))
            return GroupQuery.class;
        throw new IllegalArgumentException("Clause of unknown type in boolean query: " + booleanQuery);
    }

}
